//
// ========================================================================
// Copyright (c) 1995-2020 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under
// the terms of the Eclipse Public License 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0
//
// This Source Code may also be made available under the following
// Secondary Licenses when the conditions for such availability set
// forth in the Eclipse Public License, v. 2.0 are satisfied:
// the Apache License v2.0 which is available at
// https://www.apache.org/licenses/LICENSE-2.0
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.websocket.tests;

import java.net.URI;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import org.eclipse.jetty.client.api.AuthenticationStore;
import org.eclipse.jetty.client.util.BasicAuthentication;
import org.eclipse.jetty.security.ConstraintMapping;
import org.eclipse.jetty.security.ConstraintSecurityHandler;
import org.eclipse.jetty.security.HashLoginService;
import org.eclipse.jetty.security.SecurityHandler;
import org.eclipse.jetty.security.UserStore;
import org.eclipse.jetty.security.authentication.BasicAuthenticator;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.util.security.Constraint;
import org.eclipse.jetty.util.security.Credential;
import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.UpgradeRequest;
import org.eclipse.jetty.websocket.client.ClientUpgradeRequest;
import org.eclipse.jetty.websocket.client.WebSocketClient;
import org.eclipse.jetty.websocket.server.config.JettyWebSocketServletContainerInitializer;
import org.eclipse.jetty.websocket.tests.examples.MyAdvancedEchoServlet;
import org.eclipse.jetty.websocket.tests.examples.MyAuthedServlet;
import org.eclipse.jetty.websocket.tests.examples.MyEchoServlet;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.assertTrue;

public class WebSocketServletExamplesTest
{
    private Server _server;
    private ServerConnector connector;
    private ServletContextHandler _context;

    @BeforeEach
    public void setup() throws Exception
    {
        _server = new Server();
        connector = new ServerConnector(_server);
        _server.addConnector(connector);

        _context = new ServletContextHandler(ServletContextHandler.SESSIONS);
        _context.setContextPath("/");
        _context.setSecurityHandler(getSecurityHandler("user", "password", "testRealm"));
        _server.setHandler(_context);

        _context.addServlet(MyEchoServlet.class, "/echo");
        _context.addServlet(MyAdvancedEchoServlet.class, "/advancedEcho");
        _context.addServlet(MyAuthedServlet.class, "/authed");

        JettyWebSocketServletContainerInitializer.configure(_context, null);
        _server.start();
    }

    @AfterEach
    public void stop() throws Exception
    {
        _server.stop();
    }

    private static SecurityHandler getSecurityHandler(String username, String password, String realm)
    {

        HashLoginService loginService = new HashLoginService();
        UserStore userStore = new UserStore();
        userStore.addUser(username, Credential.getCredential(password), new String[]{"websocket"});
        loginService.setUserStore(userStore);
        loginService.setName(realm);

        Constraint constraint = new Constraint();
        constraint.setName("auth");
        constraint.setAuthenticate(true);
        constraint.setRoles(new String[]{"**"});

        ConstraintMapping mapping = new ConstraintMapping();
        mapping.setPathSpec("/authed/*");
        mapping.setConstraint(constraint);

        ConstraintSecurityHandler security = new ConstraintSecurityHandler();
        security.addConstraintMapping(mapping);
        security.setAuthenticator(new BasicAuthenticator());
        security.setLoginService(loginService);

        return security;
    }

    @Test
    public void testEchoServlet() throws Exception
    {
        WebSocketClient client = new WebSocketClient();
        client.start();

        URI uri = URI.create("ws://localhost:" + connector.getLocalPort() + "/echo");
        EventSocket socket = new EventSocket();
        CompletableFuture<Session> connect = client.connect(socket, uri);
        try (Session session = connect.get(5, TimeUnit.SECONDS))
        {
            String message = "hello world";
            session.getRemote().sendString(message);

            String response = socket.messageQueue.poll(5, TimeUnit.SECONDS);
            assertThat(response, is(message));
        }

        assertTrue(socket.closeLatch.await(10, TimeUnit.SECONDS));
    }

    @Test
    public void testAdvancedEchoServlet() throws Exception
    {
        WebSocketClient client = new WebSocketClient();
        client.start();

        URI uri = URI.create("ws://localhost:" + connector.getLocalPort() + "/advancedEcho");
        EventSocket socket = new EventSocket();

        UpgradeRequest upgradeRequest = new ClientUpgradeRequest();
        upgradeRequest.setSubProtocols("text");
        CompletableFuture<Session> connect = client.connect(socket, uri, upgradeRequest);
        try (Session session = connect.get(5, TimeUnit.SECONDS))
        {
            String message = "hello world";
            session.getRemote().sendString(message);

            String response = socket.messageQueue.poll(5, TimeUnit.SECONDS);
            assertThat(response, is(message));
        }

        assertTrue(socket.closeLatch.await(10, TimeUnit.SECONDS));
    }

    @Test
    public void testAuthedServlet() throws Exception
    {
        WebSocketClient client = new WebSocketClient();
        client.start();
        AuthenticationStore authenticationStore = client.getHttpClient().getAuthenticationStore();

        URI uri = URI.create("ws://localhost:" + connector.getLocalPort() + "/authed");

        BasicAuthentication basicAuthentication = new BasicAuthentication(uri, "testRealm", "user", "password");
        authenticationStore.addAuthentication(basicAuthentication);

        EventSocket socket = new EventSocket();
        CompletableFuture<Session> connect = client.connect(socket, uri);
        try (Session session = connect.get(5, TimeUnit.SECONDS))
        {
            String message = "hello world";
            session.getRemote().sendString(message);

            String response = socket.messageQueue.poll(5, TimeUnit.SECONDS);
            assertThat(response, is(message));
        }

        assertTrue(socket.closeLatch.await(10, TimeUnit.SECONDS));
    }
}
