#!/usr/bin/env python3
#-
# Copyright (c) 2006 Verdens Gang AS
# Copyright (c) 2006-2015 Varnish Software AS
# All rights reserved.
#
# Author: Poul-Henning Kamp <phk@phk.freebsd.dk>
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
#
# Generate various .c and .h files for the VCL compiler and the interfaces
# for it.

#######################################################################
# These are our tokens

# We could drop all words such as "include", "if" etc, and use the
# ID type instead, but declaring them tokens makes them reserved words
# which hopefully makes for better error messages.
# XXX: does it actually do that ?

import sys

srcroot = "../.."
buildroot = "../.."
if len(sys.argv) == 3:
	srcroot = sys.argv[1]
	buildroot = sys.argv[2]

tokens = {
	"T_INC":	"++",
	"T_DEC":	"--",
	"T_CAND":	"&&",
	"T_COR":	"||",
	"T_LEQ":	"<=",
	"T_EQ":		"==",
	"T_NEQ":	"!=",
	"T_GEQ":	">=",
	"T_SHR":	">>",
	"T_SHL":	"<<",
	"T_INCR":	"+=",
	"T_DECR":	"-=",
	"T_MUL":	"*=",
	"T_DIV":	"/=",
	"T_NOMATCH":	"!~",

	# Single char tokens, for convenience on one line
	None:		"{}()*+-/%><=;!&.|~,",

	# These have handwritten recognizers
	"ID":		None,
	"CNUM":		None,
	"CSTR":		None,
	"EOI":		None,
	"CSRC":		None,
}

#######################################################################
# Our methods and actions

returns =(

	###############################################################
	# Client side

	('recv',
		"C",
		('synth', 'pass', 'pipe', 'hash', 'purge',)
	),
	('pipe',
		"C",
		('synth', 'pipe',)
	),
	('pass',
		"C",
		('synth', 'restart', 'fetch',)
	),
	('hash',
		"C",
		('lookup',)
	),
	('purge',
		"C",
		('synth', 'restart',)
	),
	('miss',
		"C",
		('synth', 'restart', 'pass', 'fetch',)
	),
	('hit',
		"C",
		('synth', 'restart', 'pass', 'fetch', 'miss', 'deliver',)
	),
	('deliver',
		"C",
		('synth', 'restart', 'deliver',)
	),
	('synth',
		"C",
		('restart', 'deliver',)
	),

	###############################################################
	# Backend-fetch

	('backend_fetch',
		"B",
		('fetch', 'abandon')
	),
	('backend_response',
		"B",
		('deliver', 'retry', 'abandon')
	),
	('backend_error',
		"B",
		('deliver', 'retry', 'abandon')
	),

	###############################################################
	# Housekeeping

	('init',
		"",
		('ok', 'fail')
	),
	('fini',
		"",
		('ok',)
	),
)

#######################################################################
# Variables available in sessions
#
# 'all' means all methods
# 'client' means all methods tagged "C"
# 'backend' means all methods tagged "B"
# 'both' means all methods tagged "B" or "C"

sp_variables = [
	('remote.ip',
		'IP',
		( 'client',),
		( ), """
		The IP address of the other end of the TCP connection.
		This can either be the clients IP, or the outgoing IP
		of a proxy server.
		"""
	),
	('client.ip',
		'IP',
		( 'client',),
		( ), """
		The client's IP address.
		"""
	),
	('client.identity',
		'STRING',
		( 'client',),
		( 'client',), """
		Identification of the client, used to load balance
		in the client director.
		"""
	),
	('local.ip',
		'IP',
		( 'client',),
		( ), """
		The IP address of the local end of the TCP connection.
		"""
	),
	('server.ip',
		'IP',
		( 'client',),
		( ), """
		The IP address of the socket on which the client
		connection was received.
		"""
	),
	('server.hostname',
		'STRING',
		( 'all',),
		( ), """
		The host name of the server.
		"""
	),
	('server.identity',
		'STRING',
		( 'all',),
		( ), """
		The identity of the server, as set by the -i
		parameter.  If the -i parameter is not passed to varnishd,
		server.identity will be set to the name of the instance, as
		specified by the -n parameter.
		"""
	),
	('req',
		'HTTP',
		( 'client',),
		( ), """
		The entire request HTTP data structure
		"""
	),
	('req.method',
		'STRING',
		( 'client',),
		( 'client',), """
		The request type (e.g. "GET", "HEAD").
		"""
	),
	('req.url',
		'STRING',
		( 'client',),
		( 'client',), """
		The requested URL.
		"""
	),
	('req.proto',
		'STRING',
		( 'client',),
		( 'client',), """
		The HTTP protocol version used by the client.
		"""
	),
	('req.http.',
		'HEADER',
		( 'client',),
		( 'client',), """
		The corresponding HTTP header.
		"""
	),
	('req.restarts',
		'INT',
		( 'client',),
		( ), """
		A count of how many times this request has been restarted.
		"""
	),
	('req.esi_level',
		'INT',
		( 'client',),
		( ), """
		A count of how many levels of ESI requests we're currently at.
		"""
	),
	('req.ttl',
		'DURATION',
		( 'client',),
		( 'client',), """
		"""
	),
	('req.xid',
		'STRING',
		( 'client',),
		( ), """
		Unique ID of this request.
		"""
	),
	('req.esi',
		'BOOL',
		( 'client',),
		( 'client',), """
		Boolean. Set to false to disable ESI processing
		regardless of any value in beresp.do_esi. Defaults
		to true. This variable is subject to change in
		future versions, you should avoid using it.
		"""
	),
	('req.can_gzip',
		'BOOL',
		( 'client',),
		( ), """
		Does the client accept the gzip transfer encoding.
		"""
	),
	('req.backend_hint',
		'BACKEND',
		( 'client', ),
		( 'client',), """
		Set bereq.backend to this if we attempt to fetch.
		"""
	),
	('req.hash_ignore_busy',
		'BOOL',
		( 'recv',),
		( 'recv',), """
		Ignore any busy object during cache lookup. You
		would want to do this if you have two server looking
		up content from each other to avoid potential deadlocks.
		"""
	),
	('req.hash_always_miss',
		'BOOL',
		( 'recv',),
		( 'recv',), """
		Force a cache miss for this request. If set to true
		Varnish will disregard any existing objects and
		always (re)fetch from the backend.
		"""
	),
	('req_top.method',
		'STRING',
		( 'client',),
		(), """
		The request method of the top-level request in a tree
		of ESI requests. (e.g. "GET", "HEAD").
		Identical to req.method in non-ESI requests.
		"""
	),
	('req_top.url',
		'STRING',
		( 'client',),
		(), """
		The requested URL of the top-level request in a tree
		of ESI requests.
		Identical to req.url in non-ESI requests.
		"""
	),
	('req_top.http.',
		'HEADER',
		( 'client',),
		(), """
		HTTP headers of the top-level request in a tree of ESI requests.
		Identical to req.http. in non-ESI requests.
		"""
	),
	('req_top.proto',
		'STRING',
		( 'client',),
		(), """
		HTTP protocol version of the top-level request in a tree of
		ESI requests.
		Identical to req.proto in non-ESI requests.
		"""
	),
	('bereq',
		'HTTP',
		( 'backend',),
		( ), """
		The entire backend request HTTP data structure
		"""
	),
	('bereq.xid',
		'STRING',
		( 'backend',),
		( ), """
		Unique ID of this request.
		"""
	),
	('bereq.retries',
		'INT',
		( 'backend',),
		( ), """
		A count of how many times this request has been retried.
		"""
	),
	('bereq.backend',
		'BACKEND',
		( 'pipe', 'backend', ),
		( 'pipe', 'backend', ), """
		This is the backend or director we attempt to fetch from.
		"""
	),
	('bereq.method',
		'STRING',
		( 'pipe', 'backend', ),
		( 'pipe', 'backend', ), """
		The request type (e.g. "GET", "HEAD").
		"""
	),
	('bereq.url',
		'STRING',
		( 'pipe', 'backend', ),
		( 'pipe', 'backend', ), """
		The requested URL.
		"""
	),
	('bereq.proto',
		'STRING',
		( 'pipe', 'backend', ),
		( 'pipe', 'backend', ), """
		The HTTP protocol version used to talk to the server.
		"""
	),
	('bereq.http.',
		'HEADER',
		( 'pipe', 'backend', ),
		( 'pipe', 'backend', ), """
		The corresponding HTTP header.
		"""
	),
	('bereq.uncacheable',
		'BOOL',
		( 'backend', ),
		( ), """
		Indicates whether this request is uncacheable due
		to a pass in the client side or a hit on an existing
		uncacheable object (aka hit-for-pass).
		"""
	),
	('bereq.connect_timeout',
		'DURATION',
		( 'pipe', 'backend', ),
		( 'pipe', 'backend', ), """
		The time in seconds to wait for a backend connection.
		"""
	),
	('bereq.first_byte_timeout',
		'DURATION',
		( 'backend', ),
		( 'backend', ), """
		The time in seconds to wait for the first byte from
		the backend.  Not available in pipe mode.
		"""
	),
	('bereq.between_bytes_timeout',
		'DURATION',
		( 'backend', ),
		( 'backend', ), """
		The time in seconds to wait between each received byte from the
		backend.  Not available in pipe mode.
		"""
	),
	('beresp',
		'HTTP',
		( 'backend_response', 'backend_error'),
		( ), """
		The entire backend response HTTP data structure
		"""
	),
	('beresp.proto',
		'STRING',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		The HTTP protocol version used the backend replied with.
		"""
	),
	('beresp.status',
		'INT',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		The HTTP status code returned by the server.
		"""
	),
	('beresp.reason',
		'STRING',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		The HTTP status message returned by the server.
		"""
	),
	('beresp.http.',
		'HEADER',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		The corresponding HTTP header.
		"""
	),
	('beresp.do_esi',
		'BOOL',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		Boolean. ESI-process the object after fetching it.
		Defaults to false. Set it to true to parse the
		object for ESI directives. Will only be honored if
		req.esi is true.
		"""
	),
	('beresp.do_stream',
		'BOOL',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		Deliver the object to the client directly without
		fetching the whole object into varnish. If this
		request is pass'ed it will not be stored in memory.
		"""
	),
	('beresp.do_gzip',
		'BOOL',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		Boolean. Gzip the object before storing it. Defaults
		to false. When http_gzip_support is on Varnish will
		request already compressed content from the backend
		and as such compression in Varnish is not needed.
		"""
	),
	('beresp.do_gunzip',
		'BOOL',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		Boolean. Unzip the object before storing it in the
		cache.  Defaults to false.
		"""
	),
	('beresp.was_304',
		'BOOL',
		( 'backend_response', 'backend_error'),
		( ), """
		Boolean. If this is a successful 304 response to a
		backend conditional request refreshing an existing
		cache object.
		"""
	),
	('beresp.uncacheable',
		'BOOL',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		Inherited from bereq.uncacheable, see there.

		Setting this variable makes the object uncacheable, which may
		get stored as a hit-for-pass object in the cache.

		Clearing the variable has no effect and will log the warning
		"Ignoring attempt to reset beresp.uncacheable".
		"""
	),
	('beresp.ttl',
		'DURATION',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		The object's remaining time to live, in seconds.
		"""
	),
	('beresp.age',
		'DURATION',
		( 'backend_response', 'backend_error'),
		( ), """
		The age of the object.
		"""
	),
	('beresp.grace',
		'DURATION',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		Set to a period to enable grace.
		"""
	),
	('beresp.keep',
		'DURATION',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		Set to a period to enable conditional backend requests.

		The keep time is cache lifetime in addition to the ttl.

		Objects with ttl expired but with keep time left may be used
		to issue conditional (If-Modified-Since / If-None-Match)
		requests to the backend to refresh them.
		"""
	),
	('beresp.backend',
		'BACKEND',
		( 'backend_response', 'backend_error'),
		( ), """
		This is the backend we fetched from.  If bereq.backend
		was set to a director, this will be the backend selected
		by the director.
		"""
	),
	('beresp.backend.name',
		'STRING',
		( 'backend_response', 'backend_error'),
		( ), """
		Name of the backend this response was fetched from.
		"""
	),
	('beresp.backend.ip',
		'IP',
		( 'backend_response', 'backend_error'),
		( ), """
		IP of the backend this response was fetched from.
		"""
	),
	('beresp.storage_hint',
		'STRING',
		( 'backend_response', 'backend_error'),
		( 'backend_response', 'backend_error'), """
		Hint to Varnish that you want to save this object to a
		particular storage backend.
		"""
	),
	('obj.proto',
		'STRING',
		( 'hit', ),
		( ), """
		The HTTP protocol version used when the object was retrieved.
		"""
	),
	('obj.status',
		'INT',
		( 'hit',),
		( ), """
		The HTTP status code returned by the server.
		"""
	),
	('obj.reason',
		'STRING',
		( 'hit',),
		( ), """
		The HTTP status message returned by the server.
		"""
	),
	('obj.hits',
		'INT',
		( 'hit', 'deliver',),
		( ), """
		The count of cache-hits on this object. A value of 0 indicates a
		cache miss.
		"""
	),
	('obj.http.',
		'HEADER',
		( 'hit', ),
		( ), """
		The corresponding HTTP header.
		"""
	),
	('obj.ttl',
		'DURATION',
		( 'hit', ),
		( ), """
		The object's remaining time to live, in seconds.
		"""
	),
	('obj.age',
		'DURATION',
		( 'hit', ),
		( ), """
		The age of the object.
		"""
	),
	('obj.grace',
		'DURATION',
		( 'hit', ),
		( ), """
		The object's remaining grace period in seconds.
		"""
	),
	('obj.keep',
		'DURATION',
		( 'hit', ),
		( ), """
		The object's remaining keep period in seconds.
		"""
	),
	('obj.uncacheable',
		'BOOL',
		( 'deliver', ),
		( ), """
		Whether the object is uncacheable (pass or hit-for-pass).
		"""
	),
	('resp',
		'HTTP',
		( 'deliver', 'synth'),
		( ), """
		The entire response HTTP data structure
		"""
	),
	('resp.proto',
		'STRING',
		( 'deliver', 'synth', ),
		( 'deliver', 'synth', ), """
		The HTTP protocol version to use for the response.
		"""
	),
	('resp.status',
		'INT',
		( 'deliver', 'synth', ),
		( 'deliver', 'synth', ), """
		The HTTP status code that will be returned.
		"""
	),
	('resp.reason',
		'STRING',
		( 'deliver', 'synth', ),
		( 'deliver', 'synth', ), """
		The HTTP status message that will be returned.
		"""
	),
	('resp.http.',
		'HEADER',
		( 'deliver', 'synth', ),
		( 'deliver', 'synth', ), """
		The corresponding HTTP header.
		"""
	),
	('resp.is_streaming',
		'BOOL',
		( 'deliver', 'synth', ),
		( ), """
		Returns true when the response will be streamed
		from the backend.
		"""
	),
	('now',
		'TIME',
		( 'all',),
		( ), """
		The current time, in seconds since the epoch. When
		used in string context it returns a formatted string.
		"""
	),
]

# Backwards compatibility:
aliases = [
]

stv_variables = (
	('free_space',	'BYTES',	"0.", 'storage.<name>.free_space', """
	Free space available in the named stevedore. Only available for
	the malloc stevedore.
	"""),
	('used_space',	'BYTES',	"0.", 'storage.<name>.used_space', """
	Used space in the named stevedore. Only available for the malloc
	stevedore.
	"""),
	('happy',	'BOOL',		"0", 'storage.<name>.happy', """
	Health status for the named stevedore. Not available in any of the
	current stevedores.
	"""),
)

#######################################################################
# VCL to C type conversion

vcltypes = {
	'STRING_LIST':	"void*",
}

fi = open(srcroot + "/include/vrt.h")

for i in fi:
	j = i.split();
	if len(j) < 3:
		continue
	if j[0] != "typedef":
		continue
	if j[-1][-1] != ";":
		continue
	if j[-1][:4] != "VCL_":
		continue
	d = " ".join(j[1:-1])
	vcltypes[j[-1][4:-1]] = d
fi.close()

#######################################################################
# Nothing is easily configurable below this line.
#######################################################################

import sys
import copy

#######################################################################
# Emit a function to recognize tokens in a string

def emit_vcl_fixed_token(fo, tokens):

	recog = list()
	emit = dict()
	for i in tokens:
		j = tokens[i]
		if (j != None):
			recog.append(j)
			emit[j] = i

	recog.sort()
	rrecog = copy.copy(recog)
	rrecog.sort(key = lambda x: -len(x))

	fo.write("""
#define M1()\tdo {*q = p + 1; return (p[0]); } while (0)
#define M2(c,t)\tdo {if (p[1] == (c)) { *q = p + 2; return (t); }} while (0)

unsigned
vcl_fixed_token(const char *p, const char **q)
{

\tswitch (p[0]) {
""")
	last_initial = None
	for i in recog:
		if (i[0] == last_initial):
			continue
		last_initial = i[0]
		fo.write("\tcase '%s':\n" % last_initial)
		need_ret = True
		for j in rrecog:
			if (j[0] != last_initial):
				continue
			if len(j) == 2:
				fo.write("\t\tM2('%s', %s);\n" %
				    (j[1], emit[j]))
			elif len(j) == 1:
				fo.write("\t\tM1();\n")
				need_ret = False
			else:
				fo.write("\t\tif (")
				k = 1
				l = len(j)
				while (k < l):
					fo.write("p[%d] == '%s'" % (k, j[k]))
					fo.write(" &&")
					if (k % 3) == 0:
						fo.write("\n\t\t    ")
					else:
						fo.write(" ")
					k += 1
				fo.write("!isvar(p[%d])) {\n" % l)
				fo.write("\t\t\t*q = p + %d;\n" % l)
				fo.write("\t\t\treturn (%s);\n" % emit[j])
				fo.write("\t\t}\n")
		if need_ret:
			fo.write("\t\treturn (0);\n")
	fo.write("\tdefault:\n\t\treturn (0);\n\t}\n}\n")

#######################################################################
# Emit the vcl_tnames (token->string) conversion array

def emit_vcl_tnames(fo, tokens):
	fo.write("\nconst char * const vcl_tnames[256] = {\n")
	l = list(tokens.keys())
	l.sort()
	for i in l:
		j = tokens[i]
		if j == None:
			j = i
		if i[0] == "'":
			j = i
		fo.write("\t[%s] = \"%s\",\n" % (i, j))
	fo.write("};\n")

#######################################################################
# Read a C-source file and spit out code that outputs it with VSB_cat()

def emit_file(fo, fd, bn):
	fn = fd + "/" + bn

	fi = open(fn)
	fc = fi.read()
	fi.close()

	w = 66		# Width of lines, after white space prefix
	maxlen = 10240	# Max length of string literal

	x = 0
	l = 0
	fo.write("\n\t/* %s */\n\n" % fn)
	fo.write('\tVSB_cat(sb, "/* ---===### %s ###===--- */\\n\\n");\n' % bn)
	for c in fc:
		if l == 0:
			fo.write("\tVSB_cat(sb, \"")
			l += 12
			x += 12
		if x == 0:
			fo.write("\t    \"")
		d = c
		if c == '\n':
			d = "\\n"
		elif c == '\t':
			d = "\\t"
		elif c == '"':
			d = "\\\""
		elif c == '\\':
			d = "\\\\"

		if c == '\n' and x > w - 20:
			fo.write(d + "\"\n")
			x = 0
			continue
		if c.isspace() and x > w - 10:
			fo.write(d + "\"\n")
			x = 0
			continue

		fo.write(d)
		x += len(d)
		l += len(d)
		if l > maxlen:
			fo.write("\");\n")
			l = 0;
			x = 0
		if x > w - 3:
			fo.write("\"\n")
			x = 0
	if x != 0:
		fo.write("\"\n")
	if l != 0:
		fo.write("\t);\n")
	fo.write('\tVSB_cat(sb, "\\n");\n')

#######################################################################

def polish_tokens(tokens):
	# Expand single char tokens
	st = tokens[None]
	del tokens[None]

	for i in st:
		tokens["'" + i + "'"] = i
#######################################################################

def file_header(fo):
	fo.write("""/*
 * NB:  This file is machine generated, DO NOT EDIT!
 *
 * Edit and run generate.py instead
 */
""")

#######################################################################

polish_tokens(tokens)

fo = open(buildroot + "/lib/libvcc/vcc_token_defs.h", "w")

file_header(fo)

j = 128
l = list(tokens.keys())
l.sort()
for i in l:
	if i[0] == "'":
		continue
	fo.write("#define\t%s %d\n" % (i, j))
	j += 1
	assert j < 256

fo.close()

#######################################################################

rets = dict()
vcls = list()
vcls_client = list()
vcls_backend = list()
for i in returns:
	vcls.append(i[0])
	for j in i[1]:
		if j == "B":
			vcls_backend.append(i[0])
		elif j == "C":
			vcls_client.append(i[0])
	for j in i[2]:
		rets[j] = True

#######################################################################

fo = open(buildroot + "/include/tbl/vcl_returns.h", "w")

file_header(fo)

fo.write("\n/*lint -save -e525 -e539 */\n")

fo.write("\n#ifdef VCL_RET_MAC\n")
l = list(rets.keys())
l.sort()
ll = list(returns)
ll.sort()
for i in l:
	fo.write("VCL_RET_MAC(%s, %s" % (i.lower(), i.upper()))
	s=",\n\t"
	for j in ll:
		if i in j[2]:
			fo.write("%sVCL_MET_%s" % (s, j[0].upper()))
			s = " |\n\t"
	fo.write("\n)\n")
fo.write("#endif\n")

fo.write("\n#ifdef VCL_MET_MAC\n")
for i in ll:
	fo.write("VCL_MET_MAC(%s, %s," % (i[0].lower(), i[0].upper()))
	p = " (\n\t"
	lll = list(i[2])
	lll.sort()
	for j in lll:
		fo.write("%s(1U << VCL_RET_%s)" % (p, j.upper()))
		p = " |\n\t"
	fo.write("\n))\n")
fo.write("#endif\n")
fo.write("\n/*lint -restore */\n")
fo.close()

#######################################################################

fo = open(buildroot + "/include/vcl.h", "w")

file_header(fo)

fo.write("""
struct vrt_ctx;
#define VRT_CTX const struct vrt_ctx *ctx
struct req;
struct busyobj;
struct ws;
struct cli;
struct worker;

enum vcl_event_e {
	VCL_EVENT_LOAD,
	VCL_EVENT_WARM,
	VCL_EVENT_USE,
	VCL_EVENT_COLD,
	VCL_EVENT_DISCARD,
};

typedef int vcl_event_f(VRT_CTX, enum vcl_event_e);
typedef int vcl_init_f(VRT_CTX);
typedef void vcl_fini_f(VRT_CTX);
typedef int vcl_func_f(VRT_CTX);
""")

def tbl40(a, b):
	while len(a.expandtabs()) < 40:
		a += "\t"
	return a + b

fo.write("\n/* VCL Methods */\n")
n = 1
for i in returns:
	fo.write(
	    tbl40("#define VCL_MET_%s" % i[0].upper(), "(1U << %d)\n" % n)
	)
	n += 1

fo.write("\n" + tbl40("#define VCL_MET_MAX", "%d\n" % n))
fo.write("\n" + tbl40("#define VCL_MET_MASK", "0x%x\n" % ((1 << n) - 1)))


fo.write("\n/* VCL Returns */\n")
n = 0
l = list(rets.keys())
l.sort()
for i in l:
	fo.write(tbl40("#define VCL_RET_%s" % i.upper(), "%d\n" % n))
	n += 1

fo.write("\n" + tbl40("#define VCL_RET_MAX", "%d\n" % n))


fo.write("""
struct VCL_conf {
	unsigned			magic;
#define VCL_CONF_MAGIC			0x7406c509	/* from /dev/random */

	struct director			**default_director;
	const struct vrt_backend_probe	*default_probe;
	unsigned			nref;
	struct vrt_ref			*ref;

	unsigned			nsrc;
	const char			**srcname;
	const char			**srcbody;

	vcl_event_f			*event_vcl;
""")

for i in returns:
	fo.write("\tvcl_func_f\t*" + i[0] + "_func;\n")

fo.write("""
};
""")

fo.close()

#######################################################################

def restrict(fo, spec):
	d = dict()
	for j in spec:
		if j == 'all':
			for i in vcls:
				d[i] = True
		elif j == 'backend':
			for i in vcls_backend:
				d[i] = True
		elif j == 'client':
			for i in vcls_client:
				d[i] = True
		elif j == 'both':
			for i in vcls_client:
				d[i] = True
			for i in vcls_backend:
				d[i] = True
		else:
			assert j in vcls
			d[j] = True
	p = ""
	n = 0
	l = list(d.keys())
	l.sort()
	w = 0
	fo.write("\t\t")
	for j in l:
		x = p + "VCL_MET_" + j.upper()
		if w + len(x) > 60:
			fo.write("\n\t\t")
			w = 0
		fo.write(x)
		w += len(x)
		p = " | "
	if len(d) == 0:
		fo.write("0")
	fo.write(",\n")

#######################################################################

fh = open(buildroot + "/include/vrt_obj.h", "w")
file_header(fh)

fo = open(buildroot + "/lib/libvcc/vcc_obj.c", "w")
file_header(fo)

fo.write("""
#include "config.h"

#include <stdio.h>

#include "vcc_compile.h"

const struct var vcc_vars[] = {
""")

def one_var(nm, spec):
	fh.write("\n")
	typ = spec[1]
	cnam = i[0].replace(".", "_")
	ctyp = vcltypes[typ]

	fo.write("\t{ \"%s\", %s, %d,\n" % (nm, typ, len(nm)))

	if len(spec[2]) == 0:
		fo.write('\t    NULL,\t/* No reads allowed */\n')
	elif typ == "HEADER":
		fo.write('\t    "HDR_')
		fo.write(nm.split(".")[0].upper())
		fo.write('",\n')
	else:
		fo.write('\t    "VRT_r_%s(ctx)",\n' % cnam)
		if nm == i[0]:
			fh.write("VCL_" + typ +
			    " VRT_r_%s(VRT_CTX);\n" % cnam )
	restrict(fo, spec[2])

	if len(spec[3]) == 0:
		fo.write('\t    NULL,\t/* No writes allowed */\n')
	elif typ == "HEADER":
		fo.write('\t    "HDR_')
		fo.write(nm.split(".")[0].upper())
		fo.write('",\n')
	else:
		fo.write('\t    "VRT_l_%s(ctx, ",\n' % cnam)
		if nm == i[0]:
			fh.write(
			    "void VRT_l_%s(VRT_CTX, " % cnam)
			if typ != "STRING":
				fh.write("VCL_" + typ + ");\n")
			else:
				fh.write(ctyp + ", ...);\n")
	restrict(fo, spec[3])

	fo.write("\t},\n")


sp_variables.sort()
aliases.sort()
for i in sp_variables:
	one_var(i[0], i)
	for j in aliases:
		if j[1] == i[0]:
			one_var(j[0], i)

fo.write("\t{ NULL }\n};\n")

fh.write("\n")
for i in stv_variables:
	fh.write(vcltypes[i[1]] + " VRT_Stv_" + i[0] + "(const char *);\n")

fo.close()
fh.close()

#######################################################################

fo = open(buildroot + "/lib/libvcc/vcc_fixed_token.c", "w")

file_header(fo)
fo.write("""

#include "config.h"

#include <ctype.h>
#include <stdio.h>

#include "vcc_compile.h"
""")

emit_vcl_fixed_token(fo, tokens)
emit_vcl_tnames(fo, tokens)

fo.write("""
void
vcl_output_lang_h(struct vsb *sb)
{
""")

emit_file(fo, buildroot, "include/vdef.h")
emit_file(fo, buildroot, "include/vcl.h")
emit_file(fo, srcroot, "include/vrt.h")
emit_file(fo, buildroot, "include/vrt_obj.h")

fo.write("""
}
""")

fo.close()

#######################################################################
ft = open(buildroot + "/include/tbl/vcc_types.h", "w")
file_header(ft)

ft.write("/*lint -save -e525 -e539 */\n")

i = list(vcltypes.keys())
i.sort()
for j in i:
	ft.write("VCC_TYPE(" + j + ")\n")
ft.write("/*lint -restore */\n")
ft.close()

#######################################################################

fo = open(buildroot + "/include/tbl/vrt_stv_var.h", "w")

file_header(fo)

fo.write("""
#ifndef VRTSTVTYPE
#define VRTSTVTYPE(ct)
#define VRTSTVTYPEX
#endif
#ifndef VRTSTVVAR
#define VRTSTVVAR(nm, vtype, ctype, dval)
#define VRTSTVVARX
#endif
""")

x=dict()
for i in stv_variables:
	ct = vcltypes[i[1]]
	if not ct in x:
		fo.write("VRTSTVTYPE(" + ct + ")\n")
		x[ct] = 1
	fo.write("VRTSTVVAR(" + i[0] + ",\t" + i[1] + ",\t")
	fo.write(ct + ",\t" + i[2] + ")")
	fo.write("\n")

fo.write("""
#ifdef VRTSTVTYPEX
#undef VRTSTVTYPEX
#undef VRTSTVTYPE
#endif
#ifdef VRTSTVVARX
#undef VRTSTVVARX
#undef VRTSTVVAR
#endif
""")

fo.close

#######################################################################

fp_vclvar = open(buildroot + "/doc/sphinx/include/vcl_var.rst", "w")

l = list()
for i in sp_variables:
	l.append(i)

l.sort()

def rst_where(fo, h, l):
	ll = list()
	if len(l) == 0:
		return
	fo.write("\t" + h)
	s = ""
	for j in l:
		if j == "both":
			ll.append("client")
			ll.append("backend")
		elif j == "client":
			ll.append(j)
		elif j == "backend":
			ll.append(j)
		else:
			ll.append("vcl_" + j)
	for j in ll:
		fo.write(s + j)
		s = ", "
	fo.write("\n\n")

hdr=""
for i in l:
	j = i[0].split(".")
	if j[0] != hdr:
		fp_vclvar.write("\n" + j[0] + "\n")
		fp_vclvar.write("~" * len(j[0]) + "\n")
		hdr = j[0]
	fp_vclvar.write("\n" + i[0] + "\n\n")
	fp_vclvar.write("\tType: " + i[1] + "\n\n")
	rst_where(fp_vclvar, "Readable from: ", i[2])
	rst_where(fp_vclvar, "Writable from: ", i[3])
	for j in i[4].split("\n"):
		fp_vclvar.write("\t%s\n" % j.strip())

hdr="storage"
fp_vclvar.write("\n" + hdr + "\n");
fp_vclvar.write("~" * len(hdr) + "\n");
for i in stv_variables:
	fp_vclvar.write("\n" + i[3] + "\n\n")
	fp_vclvar.write("\tType: " + i[1] + "\n\n")
	fp_vclvar.write("\tReadable from: client, backend\n\n")
	for j in i[4].split("\n"):
		fp_vclvar.write("\t%s\n" % j.strip())


fp_vclvar.close()
