"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const process_1 = __importDefault(require("process"));
const fs_1 = __importDefault(require("fs"));
const chai_1 = require("chai");
const convert_1 = require("../src/convert");
process_1.default.on('uncaughtException', (err, origin) => {
    fs_1.default.writeSync(process_1.default.stderr.fd, `Caught exception: ${err}\n` +
        `Exception origin: ${origin}`);
    chai_1.assert.fail();
});
describe('fromQiniuPath', () => {
    it('should return correct basename', () => {
        chai_1.expect(convert_1.fromQiniuPath('file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromQiniuPath('/file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromQiniuPath('//file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromQiniuPath('dir1/').basename()).to.undefined;
        chai_1.expect(convert_1.fromQiniuPath('/dir1/').basename()).to.undefined;
        chai_1.expect(convert_1.fromQiniuPath('//dir1/').basename()).to.undefined;
        chai_1.expect(convert_1.fromQiniuPath('/dir1//').basename()).to.undefined;
        chai_1.expect(convert_1.fromQiniuPath('//dir1//').basename()).to.undefined;
        chai_1.expect(convert_1.fromQiniuPath('dir1/dir2/dir3/file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromQiniuPath('/dir1/dir2/dir3/file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromQiniuPath('/dir1//dir2/dir3/file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromQiniuPath('//dir1/dir2/dir3/file').basename()).to.equal('file');
    });
    it('should return correct directoryBasename', () => {
        chai_1.expect(convert_1.fromQiniuPath('file').directoryBasename()).to.undefined;
        chai_1.expect(convert_1.fromQiniuPath('/file').directoryBasename()).to.equal('');
        chai_1.expect(convert_1.fromQiniuPath('//file').directoryBasename()).to.equal('');
        chai_1.expect(convert_1.fromQiniuPath('dir1/').directoryBasename()).to.equal('dir1');
        chai_1.expect(convert_1.fromQiniuPath('/dir1/').directoryBasename()).to.equal('dir1');
        chai_1.expect(convert_1.fromQiniuPath('//dir1/').directoryBasename()).to.equal('dir1');
        chai_1.expect(convert_1.fromQiniuPath('/dir1//').directoryBasename()).to.equal('');
        chai_1.expect(convert_1.fromQiniuPath('//dir1//').directoryBasename()).to.equal('');
        chai_1.expect(convert_1.fromQiniuPath('dir1/dir2/dir3/file').directoryBasename()).to.equal('dir3');
        chai_1.expect(convert_1.fromQiniuPath('/dir1/dir2/dir3/file').directoryBasename()).to.equal('dir3');
        chai_1.expect(convert_1.fromQiniuPath('/dir1//dir2/dir3/file').directoryBasename()).to.equal('dir3');
        chai_1.expect(convert_1.fromQiniuPath('//dir1/dir2/dir3/file').directoryBasename()).to.equal('dir3');
    });
    it('should convert to correct string', () => {
        chai_1.expect(convert_1.fromQiniuPath('file').toString()).to.equal('file');
        chai_1.expect(convert_1.fromQiniuPath('/file').toString()).to.equal('/file');
        chai_1.expect(convert_1.fromQiniuPath('//file').toString()).to.equal('//file');
        chai_1.expect(convert_1.fromQiniuPath('dir1/').toString()).to.equal('dir1/');
        chai_1.expect(convert_1.fromQiniuPath('/dir1/').toString()).to.equal('/dir1/');
        chai_1.expect(convert_1.fromQiniuPath('//dir1/').toString()).to.equal('//dir1/');
        chai_1.expect(convert_1.fromQiniuPath('/dir1//').toString()).to.equal('/dir1//');
        chai_1.expect(convert_1.fromQiniuPath('//dir1//').toString()).to.equal('//dir1//');
        chai_1.expect(convert_1.fromQiniuPath('dir1/dir2/dir3/file').toString()).to.equal('dir1/dir2/dir3/file');
        chai_1.expect(convert_1.fromQiniuPath('/dir1/dir2/dir3/file').toString()).to.equal('/dir1/dir2/dir3/file');
        chai_1.expect(convert_1.fromQiniuPath('/dir1//dir2/dir3/file').toString()).to.equal('/dir1//dir2/dir3/file');
        chai_1.expect(convert_1.fromQiniuPath('//dir1/dir2/dir3/file').toString()).to.equal('//dir1/dir2/dir3/file');
    });
    it('should return correct parentDirectoryPath string', () => {
        chai_1.expect(convert_1.fromQiniuPath('file').parentDirectoryPath().toString()).to.equal('');
        chai_1.expect(convert_1.fromQiniuPath('/file').parentDirectoryPath().toString()).to.equal('/');
        chai_1.expect(convert_1.fromQiniuPath('//file').parentDirectoryPath().toString()).to.equal('//');
        chai_1.expect(convert_1.fromQiniuPath('dir1/').parentDirectoryPath().toString()).to.equal('');
        chai_1.expect(convert_1.fromQiniuPath('/dir1/').parentDirectoryPath().toString()).to.equal('/');
        chai_1.expect(convert_1.fromQiniuPath('//dir1/').parentDirectoryPath().toString()).to.equal('//');
        chai_1.expect(convert_1.fromQiniuPath('/dir1//').parentDirectoryPath().toString()).to.equal('/dir1/');
        chai_1.expect(convert_1.fromQiniuPath('//dir1//').parentDirectoryPath().toString()).to.equal('//dir1/');
        chai_1.expect(convert_1.fromQiniuPath('dir1/dir2/dir3/file').parentDirectoryPath().toString()).to.equal('dir1/dir2/dir3/');
        chai_1.expect(convert_1.fromQiniuPath('/dir1/dir2/dir3/file').parentDirectoryPath().toString()).to.equal('/dir1/dir2/dir3/');
        chai_1.expect(convert_1.fromQiniuPath('/dir1//dir2/dir3/file').parentDirectoryPath().toString()).to.equal('/dir1//dir2/dir3/');
        chai_1.expect(convert_1.fromQiniuPath('//dir1/dir2/dir3/file').parentDirectoryPath().toString()).to.equal('//dir1/dir2/dir3/');
    });
    it('should join folder or file', () => {
        chai_1.expect(convert_1.fromQiniuPath('/').joinFile('file').toString()).to.equal('/file');
        chai_1.expect(convert_1.fromQiniuPath('/').joinFile('file/').toString()).to.equal('/file');
        chai_1.expect(convert_1.fromQiniuPath('/').joinFolder('dir').toString()).to.equal('/dir/');
        chai_1.expect(convert_1.fromQiniuPath('/').joinFolder('dir/').toString()).to.equal('/dir/');
    });
});
describe('fromLocalPath for UNIX', () => {
    it('should return correct basename', () => {
        chai_1.expect(convert_1.fromPosixPath('file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromPosixPath('/file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromPosixPath('dir1/').basename()).to.undefined;
        chai_1.expect(convert_1.fromPosixPath('/dir1/').basename()).to.undefined;
        chai_1.expect(convert_1.fromPosixPath('dir1/dir2/dir3/file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromPosixPath('/dir1/dir2/dir3/file').basename()).to.equal('file');
    });
    it('should return correct directoryBasename', () => {
        chai_1.expect(convert_1.fromPosixPath('file').directoryBasename()).to.undefined;
        chai_1.expect(convert_1.fromPosixPath('/file').directoryBasename()).to.undefined;
        chai_1.expect(convert_1.fromPosixPath('dir1/').directoryBasename()).to.equal('dir1');
        chai_1.expect(convert_1.fromPosixPath('/dir1/').directoryBasename()).to.equal('dir1');
        chai_1.expect(convert_1.fromPosixPath('dir1/dir2/dir3/file').directoryBasename()).to.equal('dir3');
        chai_1.expect(convert_1.fromPosixPath('/dir1/dir2/dir3/file').directoryBasename()).to.equal('dir3');
    });
    it('should convert to correct string', () => {
        chai_1.expect(convert_1.fromPosixPath('file').toString()).to.equal('file');
        chai_1.expect(convert_1.fromPosixPath('/file').toString()).to.equal('/file');
        chai_1.expect(convert_1.fromPosixPath('dir1/').toString()).to.equal('dir1/');
        chai_1.expect(convert_1.fromPosixPath('/dir1/').toString()).to.equal('/dir1/');
        chai_1.expect(convert_1.fromPosixPath('dir1/dir2/dir3/file').toString()).to.equal('dir1/dir2/dir3/file');
        chai_1.expect(convert_1.fromPosixPath('/dir1/dir2/dir3/file').toString()).to.equal('/dir1/dir2/dir3/file');
    });
    it('should return correct parentDirectoryPath string', () => {
        chai_1.expect(convert_1.fromPosixPath('file').parentDirectoryPath().toString()).to.equal('');
        chai_1.expect(convert_1.fromPosixPath('/file').parentDirectoryPath().toString()).to.equal('/');
        chai_1.expect(convert_1.fromPosixPath('dir1/').parentDirectoryPath().toString()).to.equal('');
        chai_1.expect(convert_1.fromPosixPath('/dir1/').parentDirectoryPath().toString()).to.equal('/');
        chai_1.expect(convert_1.fromPosixPath('dir1/dir2/dir3/file').parentDirectoryPath().toString()).to.equal('dir1/dir2/dir3/');
        chai_1.expect(convert_1.fromPosixPath('/dir1/dir2/dir3/file').parentDirectoryPath().toString()).to.equal('/dir1/dir2/dir3/');
    });
});
describe('fromLocalPath for Windows', () => {
    it('should return correct basename', () => {
        chai_1.expect(convert_1.fromWin32Path('file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromWin32Path('C:\\file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromWin32Path('dir1\\').basename()).to.undefined;
        chai_1.expect(convert_1.fromWin32Path('C:\\dir1\\').basename()).to.undefined;
        chai_1.expect(convert_1.fromWin32Path('dir1\\dir2\\dir3\\file').basename()).to.equal('file');
        chai_1.expect(convert_1.fromWin32Path('C:\\dir1\\dir2\\dir3\\file').basename()).to.equal('file');
    });
    it('should return correct directoryBasename', () => {
        chai_1.expect(convert_1.fromWin32Path('file').directoryBasename()).to.undefined;
        chai_1.expect(convert_1.fromWin32Path('C:\\file').directoryBasename()).to.undefined;
        chai_1.expect(convert_1.fromWin32Path('dir1\\').directoryBasename()).to.equal('dir1');
        chai_1.expect(convert_1.fromWin32Path('C:\\dir1\\').directoryBasename()).to.equal('dir1');
        chai_1.expect(convert_1.fromWin32Path('dir1\\dir2\\dir3\\file').directoryBasename()).to.equal('dir3');
        chai_1.expect(convert_1.fromWin32Path('C:\\dir1\\dir2\\dir3\\file').directoryBasename()).to.equal('dir3');
    });
    it('should convert to correct string', () => {
        chai_1.expect(convert_1.fromWin32Path('file').toString()).to.equal('file');
        chai_1.expect(convert_1.fromWin32Path('C:\\file').toString()).to.equal('C:\\file');
        chai_1.expect(convert_1.fromWin32Path('dir1\\').toString()).to.equal('dir1\\');
        chai_1.expect(convert_1.fromWin32Path('C:\\dir1\\').toString()).to.equal('C:\\dir1\\');
        chai_1.expect(convert_1.fromWin32Path('dir1\\dir2\\dir3\\file').toString()).to.equal('dir1\\dir2\\dir3\\file');
        chai_1.expect(convert_1.fromWin32Path('C:\\dir1\\dir2\\dir3\\file').toString()).to.equal('C:\\dir1\\dir2\\dir3\\file');
    });
    it('should return correct parentDirectoryPath string', () => {
        chai_1.expect(convert_1.fromWin32Path('file').parentDirectoryPath().toString()).to.equal('');
        chai_1.expect(convert_1.fromWin32Path('C:\\file').parentDirectoryPath().toString()).to.equal('C:\\');
        chai_1.expect(convert_1.fromWin32Path('dir1\\').parentDirectoryPath().toString()).to.equal('');
        chai_1.expect(convert_1.fromWin32Path('C:\\dir1\\').parentDirectoryPath().toString()).to.equal('C:\\');
        chai_1.expect(convert_1.fromWin32Path('dir1\\dir2\\dir3\\file').parentDirectoryPath().toString()).to.equal('dir1\\dir2\\dir3\\');
        chai_1.expect(convert_1.fromWin32Path('C:\\dir1\\dir2\\dir3\\file').parentDirectoryPath().toString()).to.equal('C:\\dir1\\dir2\\dir3\\');
    });
});
//# sourceMappingURL=convert.js.map