"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Uploader = void 0;
const stream_throttle_1 = require("stream-throttle");
class Uploader {
    constructor(adapter) {
        this.adapter = adapter;
        this.aborted = false;
    }
    putObjectFromFile(region, object, file, fileSize, originalFileName, putFileOption) {
        this.aborted = false;
        return new Promise((resolve, reject) => {
            var _a;
            if (this.aborted) {
                reject(Uploader.userCanceledError);
                return;
            }
            const partSize = (_a = putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.partSize) !== null && _a !== void 0 ? _a : (1 << 22);
            const partsCount = partsCountOfFile(fileSize, partSize);
            if ((putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.uploadThreshold) && fileSize <= putFileOption.uploadThreshold || partsCount <= 1) {
                this.putObject(region, object, file, fileSize, originalFileName, putFileOption).then(resolve).catch(reject);
                return;
            }
            this.initParts(region, object, originalFileName, putFileOption).then((recovered) => {
                var _a;
                if (this.aborted) {
                    reject(Uploader.userCanceledError);
                    return;
                }
                if ((_a = putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.putCallback) === null || _a === void 0 ? void 0 : _a.partsInitCallback) {
                    try {
                        putFileOption.putCallback.partsInitCallback(recovered);
                    }
                    catch (err) {
                        reject(err);
                        return;
                    }
                }
                const uploaded = uploadedSizeOfParts(recovered.parts, fileSize, partSize);
                this.uploadParts(region, object, file, fileSize, uploaded, recovered, 1, partsCount, partSize, putFileOption || {}).then(() => {
                    if (this.aborted) {
                        reject(Uploader.userCanceledError);
                        return;
                    }
                    recovered.parts.sort((part1, part2) => part1.partNumber - part2.partNumber);
                    this.adapter.completeMultipartUpload(region, object, recovered.uploadId, recovered.parts, originalFileName, putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.header)
                        .then(resolve).catch(reject);
                }).catch(reject);
            }).catch(reject);
        });
    }
    abort() {
        this.aborted = true;
    }
    putObject(region, object, file, fileSize, originalFileName, putFileOption) {
        return new Promise((resolve, reject) => {
            const data = Buffer.alloc(fileSize);
            file.read(data, 0, fileSize, 0).then(({ bytesRead }) => {
                var _a, _b;
                if (this.aborted) {
                    reject(Uploader.userCanceledError);
                    return;
                }
                let throttle = undefined;
                if (putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.uploadThrottleOption) {
                    const throttleGroup = (_a = putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.uploadThrottleGroup) !== null && _a !== void 0 ? _a : new stream_throttle_1.ThrottleGroup(putFileOption.uploadThrottleOption);
                    throttle = throttleGroup.throttle(putFileOption.uploadThrottleOption);
                }
                this.adapter.putObject(region, object, data.subarray(0, bytesRead), originalFileName, putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.header, {
                    progressCallback: (_b = putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.putCallback) === null || _b === void 0 ? void 0 : _b.progressCallback,
                    throttle: throttle,
                }).then(resolve).catch(reject);
            }).catch(reject);
        });
    }
    initParts(region, object, originalFileName, putFileOption) {
        return new Promise((resolve, reject) => {
            const recovered = { uploadId: '', parts: [] };
            if ((putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.recovered) && checkParts(putFileOption.recovered.parts)) {
                recovered.uploadId = putFileOption.recovered.uploadId;
                recovered.parts = recovered.parts.concat(putFileOption.recovered.parts);
                resolve(recovered);
            }
            else {
                this.adapter.createMultipartUpload(region, object, originalFileName, putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.header).then((initPartsOutput) => {
                    recovered.uploadId = initPartsOutput.uploadId;
                    resolve(recovered);
                }).catch(reject);
            }
        });
    }
    uploadParts(region, object, file, fileSize, uploaded, recovered, partNumber, partsCount, partSize, putFileOption) {
        return new Promise((resolve, reject) => {
            if (partNumber > partsCount) {
                resolve();
                return;
            }
            if (this.aborted) {
                reject(Uploader.userCanceledError);
                return;
            }
            if (findPartsByNumber(recovered.parts, partNumber)) {
                this.uploadParts(region, object, file, fileSize, uploaded, recovered, partNumber + 1, partsCount, partSize, putFileOption)
                    .then(resolve).catch(reject);
            }
            else {
                let data = Buffer.alloc(partSize);
                file.read(data, 0, partSize, partSize * (partNumber - 1)).then(({ bytesRead }) => {
                    var _a;
                    if (this.aborted) {
                        reject(Uploader.userCanceledError);
                        return;
                    }
                    const makeThrottle = () => {
                        if (putFileOption.uploadThrottleOption) {
                            if (!putFileOption.uploadThrottleGroup) {
                                putFileOption.uploadThrottleGroup = new stream_throttle_1.ThrottleGroup(putFileOption.uploadThrottleOption);
                            }
                            return putFileOption.uploadThrottleGroup.throttle(putFileOption.uploadThrottleOption);
                        }
                        return undefined;
                    };
                    let progressCallback = undefined;
                    if ((_a = putFileOption.putCallback) === null || _a === void 0 ? void 0 : _a.progressCallback) {
                        progressCallback = (partUploaded, _partTotal) => {
                            putFileOption.putCallback.progressCallback(uploaded + partUploaded, fileSize);
                        };
                    }
                    this.adapter.uploadPart(region, object, recovered.uploadId, partNumber, data.subarray(0, bytesRead), {
                        progressCallback: progressCallback,
                        throttle: makeThrottle(),
                    }).then((output) => {
                        var _a;
                        data = undefined;
                        const part = { etag: output.etag, partNumber: partNumber };
                        if ((_a = putFileOption === null || putFileOption === void 0 ? void 0 : putFileOption.putCallback) === null || _a === void 0 ? void 0 : _a.partPutCallback) {
                            try {
                                putFileOption.putCallback.partPutCallback(part);
                            }
                            catch (err) {
                                reject(err);
                                return;
                            }
                        }
                        recovered.parts.push(part);
                        uploaded += bytesRead;
                        this.uploadParts(region, object, file, fileSize, uploaded, recovered, partNumber + 1, partsCount, partSize, putFileOption)
                            .then(resolve).catch(reject);
                    }).catch(reject);
                }).catch(reject);
            }
        });
    }
}
exports.Uploader = Uploader;
Uploader.userCanceledError = new Error('User Canceled');
function checkParts(parts) {
    const partNumbers = new Set();
    for (const part of parts) {
        partNumbers.add(part.partNumber);
    }
    return partNumbers.size === parts.length;
}
function findPartsByNumber(parts, partNumber) {
    return parts.find((part) => part.partNumber === partNumber);
}
function partsCountOfFile(fileSize, partSize) {
    const count = (fileSize + partSize - 1) / partSize;
    return ~~count;
}
function uploadedSizeOfParts(parts, fileSize, partSize) {
    const partsCount = partsCountOfFile(fileSize, partSize);
    let uploaded = 0;
    parts.forEach((part) => {
        uploaded += partSize;
        if (part.partNumber === partsCount) {
            uploaded -= (partSize * partsCount - fileSize);
        }
    });
    return uploaded;
}
//# sourceMappingURL=uploader.js.map