/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.start;

import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.jetty.start.BaseHome;
import org.eclipse.jetty.start.FS;
import org.eclipse.jetty.start.Module;
import org.eclipse.jetty.start.StartArgs;
import org.eclipse.jetty.start.StartLog;
import org.eclipse.jetty.start.UsageException;
import org.eclipse.jetty.start.shaded.util.TopologicalSort;

public class Modules
implements Iterable<Module> {
    private final List<Module> _modules = new ArrayList<Module>();
    private final Map<String, Module> _names = new HashMap<String, Module>();
    private final Map<String, Set<Module>> _provided = new HashMap<String, Set<Module>>();
    private final Map<String, String> _providedDefaults = new HashMap<String, String>();
    private final BaseHome _baseHome;
    private final StartArgs _args;
    private final Properties _deprecated = new Properties();

    public Modules(BaseHome basehome, StartArgs args) {
        String javaVersion;
        this._baseHome = basehome;
        this._args = args;
        if (!args.getProperties().containsKey("java.version") && (javaVersion = System.getProperty("java.version")) != null) {
            args.setProperty("java.version", javaVersion, "<internal>");
        }
        try {
            Path deprecatedPath = this._baseHome.getPath("modules/deprecated.properties");
            if (deprecatedPath != null && FS.exists(deprecatedPath)) {
                this._deprecated.load(new FileInputStream(deprecatedPath.toFile()));
            }
        }
        catch (IOException e) {
            StartLog.debug(e);
        }
    }

    public void showModules(List<String> modules) {
        Stream stream = modules.contains("*") || modules.isEmpty() ? this._modules.stream().sorted() : modules.stream().map(this::get);
        stream.forEach(module -> {
            String label;
            if (module == null) {
                return;
            }
            Set<String> provides = module.getProvides();
            provides.remove(module.getName());
            System.out.printf("%n     Module: %s %s%n", module.getName(), provides.size() > 0 ? provides : "");
            for (String description : module.getDescription()) {
                System.out.printf("           : %s%n", description);
            }
            if (!module.getTags().isEmpty()) {
                label = "       Tags: %s";
                for (String t : module.getTags()) {
                    System.out.printf(label, t);
                    label = ", %s";
                }
                System.out.println();
            }
            if (!module.getDepends().isEmpty()) {
                label = "     Depend: %s";
                for (String parent : module.getDepends()) {
                    parent = Module.normalizeModuleName(parent);
                    System.out.printf(label, parent);
                    if (Module.isConditionalDependency(parent)) {
                        System.out.print(" [conditional]");
                    }
                    label = ", %s";
                }
                System.out.println();
            }
            if (!module.getOptional().isEmpty()) {
                label = "   Optional: %s";
                for (String parent : module.getOptional()) {
                    System.out.printf(label, parent);
                    label = ", %s";
                }
                System.out.println();
            }
            for (String lib : module.getLibs()) {
                System.out.printf("        LIB: %s%n", lib);
            }
            for (String xml : module.getXmls()) {
                System.out.printf("        XML: %s%n", xml);
            }
            for (String jpms : module.getJPMS()) {
                System.out.printf("        JPMS: %s%n", jpms);
            }
            for (String jvm : module.getJvmArgs()) {
                System.out.printf("        JVM: %s%n", jvm);
            }
            if (module.isEnabled()) {
                for (String selection : module.getEnableSources()) {
                    System.out.printf("    Enabled: %s%n", selection);
                }
            }
        });
    }

    public void listModules(List<String> tags) {
        if (tags.contains("-*")) {
            return;
        }
        tags = new ArrayList<String>(tags);
        boolean wild = tags.contains("*");
        HashSet included = new HashSet();
        if (wild) {
            tags.remove("*");
        } else {
            tags.stream().filter(t -> !t.startsWith("-")).forEach(included::add);
        }
        HashSet<String> excluded = new HashSet<String>();
        tags.stream().filter(t -> t.startsWith("-")).map(t -> t.substring(1)).forEach(excluded::add);
        if (!included.contains("internal")) {
            excluded.add("internal");
        }
        Predicate<Module> filter = m -> {
            if (!included.isEmpty()) {
                if (!m.getTags().stream().anyMatch(included::contains)) return false;
            }
            if (m.getTags().stream().anyMatch(excluded::contains)) return false;
            return true;
        };
        Optional<Integer> max = this._modules.stream().filter(filter).map(Module::getName).map(String::length).max(Integer::compareTo);
        if (max.isEmpty()) {
            return;
        }
        String format = "%" + max.get() + "s - %s%n";
        Comparator<Module> comparator = wild ? Comparator.comparing(Module::getName) : Module::compareTo;
        AtomicReference tag = new AtomicReference();
        this._modules.stream().filter(filter).sorted(comparator).forEach(module -> {
            if (!wild && !module.getPrimaryTag().equals(tag.get())) {
                tag.set(module.getPrimaryTag());
                System.out.printf("%n%s modules:", module.getPrimaryTag());
                System.out.printf("%n%s---------%n", "-".repeat(module.getPrimaryTag().length()));
            }
            List<String> description = module.getDescription();
            System.out.printf(format, module.getName(), description != null && description.size() > 0 ? description.get(0) : "");
        });
    }

    public void listEnabled() {
        System.out.println();
        System.out.println("Enabled Modules:");
        System.out.println("----------------");
        int i = 0;
        List<Module> enabled = this.getEnabled();
        for (Module module : enabled) {
            String name = module.getName();
            Object index = i++ + ")";
            for (String s : module.getEnableSources()) {
                System.out.printf("  %4s %-15s %s%n", index, name, s);
                index = "";
                name = "";
            }
            if (!module.isTransitive() || !module.hasIniTemplate()) continue;
            System.out.printf("                       init template available with --add-module=%s%n", module.getName());
        }
    }

    public void registerAll() throws IOException {
        for (Path path : this._baseHome.getPaths("modules/*.mod")) {
            this.registerModule(path);
        }
    }

    private Module registerModule(Path file) {
        if (!FS.canReadFile(file)) {
            throw new IllegalStateException("Cannot read file: " + file);
        }
        String shortName = this._baseHome.toShortForm(file);
        try {
            StartLog.debug("Registering Module: %s", shortName);
            Module module = new Module(this._baseHome, file);
            this._modules.add(module);
            this._names.put(module.getName(), module);
            module.getProvides().forEach(n -> {
                String name = n;
                boolean isDefaultProvider = false;
                int idx = n.indexOf(124);
                if (idx > 0) {
                    name = n.substring(0, idx);
                    isDefaultProvider = n.substring(idx + 1).equalsIgnoreCase("default");
                }
                this._provided.computeIfAbsent(name, k -> new HashSet()).add(module);
                if (isDefaultProvider) {
                    this._providedDefaults.computeIfAbsent(name, k -> module.getName());
                }
            });
            return module;
        }
        catch (Error | RuntimeException t) {
            throw t;
        }
        catch (Throwable t) {
            throw new IllegalStateException("Unable to register module: " + shortName, t);
        }
    }

    public String toString() {
        StringBuilder str = new StringBuilder();
        str.append("Modules[");
        str.append("count=").append(this._modules.size());
        str.append(",<");
        AtomicBoolean delim = new AtomicBoolean(false);
        this._modules.forEach(m -> {
            if (delim.get()) {
                str.append(',');
            }
            str.append(m.getName());
            delim.set(true);
        });
        str.append(">");
        str.append("]");
        return str.toString();
    }

    public List<Module> getEnabled() {
        List<Module> enabled = this._modules.stream().filter(Module::isEnabled).collect(Collectors.toList());
        TopologicalSort<Module> sort = new TopologicalSort<Module>();
        for (Module module : enabled) {
            Consumer<String> add = name -> {
                Set<Module> provided;
                Module dependency = this._names.get(name);
                if (dependency != null && dependency.isEnabled()) {
                    sort.addDependency(module, dependency);
                }
                if ((provided = this._provided.get(name)) != null) {
                    for (Module p : provided) {
                        if (!p.isEnabled()) continue;
                        sort.addDependency(module, p);
                    }
                }
            };
            module.getDepends().forEach(add);
            module.getOptional().forEach(add);
        }
        sort.sort(enabled);
        return enabled;
    }

    public List<Module> getSortedAll() {
        ArrayList<Module> all = new ArrayList<Module>(this._modules);
        TopologicalSort<Module> sort = new TopologicalSort<Module>();
        for (Module module : all) {
            Consumer<String> add = name -> {
                Set<Module> provided;
                Module dependency = this._names.get(name);
                if (dependency != null) {
                    sort.addDependency(module, dependency);
                }
                if ((provided = this._provided.get(name)) != null) {
                    for (Module p : provided) {
                        sort.addDependency(module, p);
                    }
                }
            };
            module.getDepends().forEach(add);
            module.getOptional().forEach(add);
        }
        sort.sort(all);
        return all;
    }

    public List<String> getSortedNames(List<String> enabledModules) {
        List<Module> all = this.getSortedAll();
        ArrayList<String> order = new ArrayList<String>();
        for (Module module : all) {
            String name = module.getName();
            if (!enabledModules.contains(name)) continue;
            order.add(name);
        }
        return order;
    }

    public Set<String> enable(String name, String enabledFrom) {
        Module module = this.get(name);
        if (module == null) {
            throw new UsageException(-9, "Unknown module='%s'. List available with --list-modules", name);
        }
        HashSet<String> enabled = new HashSet<String>();
        this.enable(enabled, module, enabledFrom, false);
        return enabled;
    }

    private void enable(Set<String> newlyEnabled, Module module, String enabledFrom, boolean transitive) {
        StartLog.debug("Enable [%s] from [%s] transitive=%b", module, enabledFrom, transitive);
        if (newlyEnabled.contains(module.getName())) {
            StartLog.debug("Already enabled [%s] from %s", module.getName(), module.getEnableSources());
            return;
        }
        for (String name : module.getProvides()) {
            Set<Module> providers = this._provided.get(name);
            if (providers == null) continue;
            for (Module p : providers) {
                if (p.equals(module) || !p.isEnabled()) continue;
                if (p.isTransitive() && !transitive) {
                    p.clearTransitiveEnable();
                    continue;
                }
                throw new UsageException("Module %s provides %s, which is already provided by %s enabled in %s", module.getName(), name, p.getName(), p.getEnableSources());
            }
        }
        if (module.enable(enabledFrom, transitive)) {
            StartLog.debug("Enabled [%s]", module.getName());
            newlyEnabled.add(module.getName());
            module.expandDependencies(this._args.getProperties());
            if (module.hasDefaultConfig()) {
                for (String line : module.getDefaultConfig()) {
                    this._args.parse(line, module.getName() + "[ini]");
                }
                for (Module m2 : this._modules) {
                    m2.expandDependencies(this._args.getProperties());
                }
            }
        }
        StartLog.debug("Enabled module [%s] depends on %s", module.getName(), module.getDepends());
        for (String dependsOnRaw : module.getDepends()) {
            boolean isConditional = Module.isConditionalDependency(dependsOnRaw);
            String dependentModule = Module.normalizeModuleName(dependsOnRaw);
            Set<Module> providers = this.getAvailableProviders(dependentModule);
            StartLog.debug("Module [%s] depends on [%s] provided by %s", module, dependentModule, providers);
            if (providers.isEmpty()) {
                if (dependentModule.contains("/")) {
                    Path file = this._baseHome.getPath("modules/" + dependentModule + ".mod");
                    if (!isConditional || Files.exists(file, new LinkOption[0])) {
                        this.registerModule(file).expandDependencies(this._args.getProperties());
                        providers = this._provided.get(dependentModule);
                        if (providers == null || providers.isEmpty()) {
                            throw new UsageException("Module %s does not provide %s", this._baseHome.toShortForm(file), dependentModule);
                        }
                        this.enable(newlyEnabled, (Module)providers.stream().findFirst().get(), "dynamic dependency of " + module.getName(), true);
                        continue;
                    }
                }
                if (isConditional) {
                    StartLog.debug("Skipping conditional module [%s]: it does not exist", dependentModule);
                    continue;
                }
                throw new UsageException("No module found to provide %s for %s", dependentModule, module);
            }
            if (providers.stream().anyMatch(Module::isEnabled)) {
                providers.stream().filter(m -> m.isEnabled() && !m.equals(module)).forEach(m -> this.enable(newlyEnabled, (Module)m, "transitive provider of " + dependentModule + " for " + module.getName(), true));
                continue;
            }
            Optional<Module> dftProvider = this.findDefaultProvider(providers, dependentModule);
            if (!dftProvider.isPresent()) continue;
            StartLog.debug("Using [%s] provider as default for [%s]", dftProvider.get(), dependentModule);
            this.enable(newlyEnabled, dftProvider.get(), "transitive provider of " + dependentModule + " for " + module.getName(), true);
        }
    }

    private Optional<Module> findDefaultProvider(Set<Module> providers, String dependsOn) {
        if (providers.size() == 1) {
            return providers.stream().findFirst();
        }
        if (providers.size() > 1) {
            String defaultProviderName = this._providedDefaults.get(dependsOn);
            if (defaultProviderName != null) {
                return providers.stream().filter(m -> m.getName().equals(defaultProviderName)).findFirst();
            }
            return providers.stream().filter(m -> m.getName().equals(dependsOn)).findFirst();
        }
        return Optional.empty();
    }

    private Set<Module> getAvailableProviders(String name) {
        Set<Module> providers = this._provided.get(name);
        StartLog.debug("Providers of [%s] are %s", name, providers);
        if (providers == null || providers.isEmpty()) {
            return Collections.emptySet();
        }
        providers = new HashSet<Module>(providers);
        HashSet<String> provided = new HashSet<String>();
        for (Module m : this._modules) {
            if (!m.isEnabled()) continue;
            provided.add(m.getName());
            provided.addAll(m.getProvides());
        }
        Iterator<Module> i = providers.iterator();
        block1: while (i.hasNext()) {
            Module provider = i.next();
            if (provider.isEnabled()) continue;
            for (String p : provider.getProvides()) {
                if (!provided.contains(p)) continue;
                StartLog.debug("Removing provider %s because %s already enabled", provider, p);
                i.remove();
                continue block1;
            }
        }
        StartLog.debug("Available providers of [%s] are %s", name, providers);
        return providers;
    }

    public Module get(String name) {
        String reason;
        Module module = this._names.get(name);
        if (module == null && (reason = this._deprecated.getProperty(name)) != null) {
            StartLog.warn("Module %s is no longer available: %s", name, reason);
        }
        return module;
    }

    @Override
    public Iterator<Module> iterator() {
        return this._modules.iterator();
    }

    public Stream<Module> stream() {
        return this._modules.stream();
    }

    public void checkEnabledModules() {
        StringBuilder unsatisfied = new StringBuilder();
        this._modules.stream().filter(Module::isEnabled).forEach(m -> m.getDepends().stream().filter(depends -> !Module.isConditionalDependency(depends)).forEach(d -> {
            Set<Module> providers = this.getAvailableProviders((String)d);
            if (providers.stream().noneMatch(Module::isEnabled)) {
                if (unsatisfied.length() > 0) {
                    unsatisfied.append(',');
                }
                unsatisfied.append(m.getName());
                StartLog.error("Module [%s] requires a module providing [%s] from one of %s%n", m.getName(), d, providers);
            }
        }));
        if (unsatisfied.length() > 0) {
            throw new UsageException(-1, "Unsatisfied module dependencies: " + unsatisfied);
        }
    }
}

