"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const process_1 = __importDefault(require("process"));
const fs_1 = __importDefault(require("fs"));
const chai_1 = require("chai");
const uplog_1 = require("../uplog");
const fsPromises = fs_1.default.promises;
process_1.default.on('uncaughtException', (err, origin) => {
    fs_1.default.writeSync(process_1.default.stderr.fd, `Caught exception: ${err}\n` +
        `Exception origin: ${origin}`);
    chai_1.assert.fail();
});
describe('UplogBuffer', () => {
    context('Uplog buffer write', () => {
        it('writes multiple logs into buffer and read them out', async () => {
            await fsPromises.truncate(uplog_1.UplogBufferFilePath);
            let onBufferFullCallbackTimes = 0, buffer = '';
            const uplogBuffer = new uplog_1.UplogBuffer({
                appName: "fakeAppName", appVersion: "fakeAppVersion", bufferSize: 100,
                onBufferFull: (chunk) => {
                    buffer += chunk.toString();
                    onBufferFullCallbackTimes += 1;
                    return Promise.resolve();
                },
            });
            const logRequest1 = uplogBuffer.log({
                log_type: uplog_1.LogType.Request,
                status_code: 200,
                host: "fake.host.1",
                port: 80,
                method: "GET",
                total_elapsed_time: 20,
            });
            const logRequest2 = uplogBuffer.log({
                log_type: uplog_1.LogType.Request,
                status_code: 200,
                host: "fake.host.2",
                port: 80,
                method: "GET",
                total_elapsed_time: 21,
            });
            const logRequest3 = uplogBuffer.log({
                log_type: uplog_1.LogType.Request,
                status_code: 200,
                host: "fake.host.3",
                port: 80,
                method: "GET",
                total_elapsed_time: 21,
            });
            const logRequest4 = uplogBuffer.log({
                log_type: uplog_1.LogType.Request,
                status_code: 200,
                host: "fake.host.4",
                port: 80,
                method: "GET",
                total_elapsed_time: 21,
            });
            await Promise.all([logRequest1, logRequest2, logRequest3, logRequest4]);
            const uplogBufferFileStat = await fsPromises.stat(uplog_1.UplogBufferFilePath);
            chai_1.expect(uplogBufferFileStat.size).to.equal(0);
            chai_1.expect(onBufferFullCallbackTimes).to.equal(1);
            const logEntries = buffer.split("\n").filter(line => line.length > 0).map(line => JSON.parse(line));
            chai_1.expect(logEntries).to.have.lengthOf(4);
            chai_1.expect(logEntries[0].log_type).to.equal('request');
            chai_1.expect(logEntries[0].status_code).to.equal(200);
            chai_1.expect(logEntries[0].host).to.equal('fake.host.1');
            chai_1.expect(logEntries[0].sdk_name).to.equal('fakeAppName');
            chai_1.expect(logEntries[0].sdk_version).to.equal('fakeAppVersion');
            chai_1.expect(logEntries[0]).to.have.property('up_time');
            chai_1.expect(logEntries[1].log_type).to.equal('request');
            chai_1.expect(logEntries[1].status_code).to.equal(200);
            chai_1.expect(logEntries[1].host).to.equal('fake.host.2');
            chai_1.expect(logEntries[1].sdk_name).to.equal('fakeAppName');
            chai_1.expect(logEntries[1].sdk_version).to.equal('fakeAppVersion');
            chai_1.expect(logEntries[1]).to.have.property('up_time');
            chai_1.expect(logEntries[2].log_type).to.equal('request');
            chai_1.expect(logEntries[2].status_code).to.equal(200);
            chai_1.expect(logEntries[2].host).to.equal('fake.host.3');
            chai_1.expect(logEntries[2].sdk_name).to.equal('fakeAppName');
            chai_1.expect(logEntries[2].sdk_version).to.equal('fakeAppVersion');
            chai_1.expect(logEntries[2]).to.have.property('up_time');
            chai_1.expect(logEntries[3].log_type).to.equal('request');
            chai_1.expect(logEntries[3].status_code).to.equal(200);
            chai_1.expect(logEntries[3].host).to.equal('fake.host.4');
            chai_1.expect(logEntries[3].sdk_name).to.equal('fakeAppName');
            chai_1.expect(logEntries[3].sdk_version).to.equal('fakeAppVersion');
            chai_1.expect(logEntries[3]).to.have.property('up_time');
        });
    });
});
//# sourceMappingURL=uplog.js.map