/*
 * Decompiled with CFR 0.152.
 */
package jenkins.model;

import hudson.Extension;
import hudson.Util;
import hudson.model.RootAction;
import hudson.util.AtomicFileWriter;
import hudson.util.StreamTaskListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import jenkins.model.Jenkins;
import org.apache.commons.io.Charsets;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.time.FastDateFormat;
import org.apache.tools.ant.BuildException;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;
import org.kohsuke.stapler.StaplerProxy;
import org.kohsuke.stapler.framework.io.WriterOutputStream;

@Restricted(value={NoExternalUse.class})
public final class RunIdMigrator {
    private final DateFormat legacyIdFormatter = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss");
    static final Logger LOGGER = Logger.getLogger(RunIdMigrator.class.getName());
    private static final String MAP_FILE = "legacyIds";
    private static final Map<String, Integer> EMPTY = new TreeMap<String, Integer>();
    private static final Set<File> offeredToUnmigrate = Collections.synchronizedSet(new HashSet());
    @Nonnull
    private Map<String, Integer> idToNumber = EMPTY;
    private static final Pattern NUMBER_ELT = Pattern.compile("(?m)^  <number>(\\d+)</number>(\r?\n)");
    private static final Pattern ID_ELT = Pattern.compile("(?m)^  <id>([0-9_-]+)</id>(\r?\n)");
    private static final Pattern TIMESTAMP_ELT = Pattern.compile("(?m)^  <timestamp>(\\d+)</timestamp>(\r?\n)");

    private boolean load(File dir) {
        File f = new File(dir, MAP_FILE);
        if (!f.isFile()) {
            return false;
        }
        if (f.length() == 0L) {
            return true;
        }
        this.idToNumber = new TreeMap<String, Integer>();
        try {
            for (String line : FileUtils.readLines((File)f)) {
                int i = line.indexOf(32);
                this.idToNumber.put(line.substring(0, i), Integer.parseInt(line.substring(i + 1)));
            }
        }
        catch (Exception x) {
            LOGGER.log(Level.WARNING, "could not read from " + f, x);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void save(File dir) {
        File f = new File(dir, MAP_FILE);
        try {
            AtomicFileWriter w = new AtomicFileWriter(f);
            try {
                for (Map.Entry<String, Integer> entry : this.idToNumber.entrySet()) {
                    w.write(entry.getKey() + ' ' + entry.getValue() + '\n');
                }
                w.commit();
            }
            finally {
                w.abort();
            }
        }
        catch (IOException x) {
            LOGGER.log(Level.WARNING, "could not save changes to " + f, x);
        }
    }

    public void created(File dir) {
        this.save(dir);
    }

    public synchronized boolean migrate(File dir, @CheckForNull File jenkinsHome) {
        if (this.load(dir)) {
            LOGGER.log(Level.FINER, "migration already performed for {0}", dir);
            return false;
        }
        if (!dir.isDirectory()) {
            LOGGER.log(Level.FINE, "{0} was unexpectedly missing", dir);
            return false;
        }
        LOGGER.log(Level.INFO, "Migrating build records in {0}", dir);
        this.doMigrate(dir);
        this.save(dir);
        if (jenkinsHome != null && offeredToUnmigrate.add(jenkinsHome)) {
            LOGGER.log(Level.WARNING, "Build record migration (https://wiki.jenkins-ci.org/display/JENKINS/JENKINS-24380+Migration) is one-way. If you need to downgrade Jenkins, run: {0}", RunIdMigrator.getUnmigrationCommandLine(jenkinsHome));
        }
        return true;
    }

    private static String getUnmigrationCommandLine(File jenkinsHome) {
        StringBuilder cp = new StringBuilder();
        for (Class c : new Class[]{RunIdMigrator.class, Charsets.class, WriterOutputStream.class, BuildException.class, FastDateFormat.class}) {
            URL location = c.getProtectionDomain().getCodeSource().getLocation();
            String locationS = location.toString();
            if (location.getProtocol().equals("file")) {
                try {
                    locationS = new File(location.toURI()).getAbsolutePath();
                }
                catch (URISyntaxException x) {
                    // empty catch block
                }
            }
            if (cp.length() > 0) {
                cp.append(File.pathSeparator);
            }
            cp.append(locationS);
        }
        return String.format("java -classpath \"%s\" %s \"%s\"", cp, RunIdMigrator.class.getName(), jenkinsHome);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doMigrate(File dir) {
        String name;
        this.idToNumber = new TreeMap<String, Integer>();
        File[] kids = dir.listFiles();
        ArrayList<File> kidsList = new ArrayList<File>(Arrays.asList(kids));
        Iterator it = kidsList.iterator();
        while (it.hasNext()) {
            File kid = (File)it.next();
            name = kid.getName();
            try {
                Integer.parseInt(name);
            }
            catch (NumberFormatException x) {
                LOGGER.log(Level.FINE, "ignoring nonnumeric entry {0}", name);
                continue;
            }
            try {
                if (Util.isSymlink(kid)) {
                    LOGGER.log(Level.FINE, "deleting build number symlink {0} \u2192 {1}", new Object[]{name, Util.resolveSymlink(kid)});
                } else {
                    if (kid.isDirectory()) {
                        LOGGER.log(Level.FINE, "ignoring build directory {0}", name);
                        continue;
                    }
                    LOGGER.log(Level.WARNING, "need to delete anomalous file entry {0}", name);
                }
                Util.deleteFile(kid);
                it.remove();
            }
            catch (Exception x) {
                LOGGER.log(Level.WARNING, "failed to process " + kid, x);
            }
        }
        for (File kid : kidsList) {
            try {
                name = kid.getName();
                try {
                    Integer.parseInt(name);
                    LOGGER.log(Level.FINE, "skipping new build dir {0}", name);
                }
                catch (NumberFormatException x) {
                    long timestamp;
                    if (!kid.isDirectory()) {
                        LOGGER.log(Level.FINE, "skipping non-directory {0}", name);
                        continue;
                    }
                    try {
                        DateFormat dateFormat = this.legacyIdFormatter;
                        synchronized (dateFormat) {
                            timestamp = this.legacyIdFormatter.parse(name).getTime();
                        }
                    }
                    catch (ParseException x2) {
                        LOGGER.log(Level.WARNING, "found unexpected dir {0}", name);
                        continue;
                    }
                    File buildXml = new File(kid, "build.xml");
                    if (!buildXml.isFile()) {
                        LOGGER.log(Level.WARNING, "found no build.xml in {0}", name);
                        continue;
                    }
                    String xml = FileUtils.readFileToString((File)buildXml, (Charset)Charsets.UTF_8);
                    Matcher m = NUMBER_ELT.matcher(xml);
                    if (!m.find()) {
                        LOGGER.log(Level.WARNING, "could not find <number> in {0}/build.xml", name);
                        continue;
                    }
                    int number = Integer.parseInt(m.group(1));
                    String nl = m.group(2);
                    xml = m.replaceFirst("  <id>" + name + "</id>" + nl + "  <timestamp>" + timestamp + "</timestamp>" + nl);
                    File newKid = new File(dir, Integer.toString(number));
                    RunIdMigrator.move(kid, newKid);
                    FileUtils.writeStringToFile((File)new File(newKid, "build.xml"), (String)xml, (Charset)Charsets.UTF_8);
                    LOGGER.log(Level.FINE, "fully processed {0} \u2192 {1}", new Object[]{name, number});
                    this.idToNumber.put(name, number);
                }
            }
            catch (Exception x) {
                LOGGER.log(Level.WARNING, "failed to process " + kid, x);
            }
        }
    }

    static void move(File src, File dest) throws IOException {
        try {
            Files.move(src.toPath(), dest.toPath(), new CopyOption[0]);
        }
        catch (IOException x) {
            throw x;
        }
        catch (Exception x) {
            throw new IOException(x);
        }
    }

    public synchronized int findNumber(@Nonnull String id) {
        Integer number = this.idToNumber.get(id);
        return number != null ? number : 0;
    }

    public synchronized void delete(File dir, String id) {
        if (this.idToNumber.remove(id) != null) {
            this.save(dir);
        }
    }

    public static void main(String ... args) throws Exception {
        if (args.length != 1) {
            throw new Exception("pass one parameter, $JENKINS_HOME");
        }
        File root = new File(args[0]);
        File jobs = new File(root, "jobs");
        if (!jobs.isDirectory()) {
            throw new FileNotFoundException("no such $JENKINS_HOME " + root);
        }
        new RunIdMigrator().unmigrateJobsDir(jobs);
    }

    private void unmigrateJobsDir(File jobs) throws Exception {
        File[] jobDirs = jobs.listFiles();
        if (jobDirs == null) {
            System.err.println(jobs + " claimed to exist, but cannot be listed");
            return;
        }
        for (File job : jobDirs) {
            File[] kids;
            if (job.getName().equals("builds")) {
                this.unmigrateBuildsDir(job);
            }
            if ((kids = job.listFiles()) == null) continue;
            for (File kid : kids) {
                if (!kid.isDirectory()) continue;
                if (kid.getName().equals("builds")) {
                    this.unmigrateBuildsDir(kid);
                    continue;
                }
                this.unmigrateJobsDir(kid);
            }
        }
    }

    private void unmigrateBuildsDir(File builds) throws Exception {
        File mapFile = new File(builds, MAP_FILE);
        if (!mapFile.isFile()) {
            System.err.println(builds + " does not look to have been migrated yet; skipping");
            return;
        }
        for (File build : builds.listFiles()) {
            String id;
            int number;
            try {
                number = Integer.parseInt(build.getName());
            }
            catch (NumberFormatException x) {
                continue;
            }
            File buildXml = new File(build, "build.xml");
            if (!buildXml.isFile()) {
                System.err.println(buildXml + " did not exist");
                continue;
            }
            String xml = FileUtils.readFileToString((File)buildXml, (Charset)Charsets.UTF_8);
            Matcher m = TIMESTAMP_ELT.matcher(xml);
            if (!m.find()) {
                System.err.println(buildXml + " did not contain <timestamp> as expected");
                continue;
            }
            long timestamp = Long.parseLong(m.group(1));
            String nl = m.group(2);
            xml = m.replaceFirst("  <number>" + number + "</number>" + nl);
            if ((m = ID_ELT.matcher(xml)).find()) {
                id = m.group(1);
                xml = m.replaceFirst("");
            } else {
                id = this.legacyIdFormatter.format(new Date(timestamp));
            }
            FileUtils.write((File)buildXml, (CharSequence)xml, (Charset)Charsets.UTF_8);
            if (!build.renameTo(new File(builds, id))) {
                System.err.println(build + " could not be renamed");
            }
            Util.createSymlink(builds, id, Integer.toString(number), StreamTaskListener.fromStderr());
        }
        Util.deleteFile(mapFile);
        System.err.println(builds + " has been restored to its original format");
    }

    @Extension
    public static class UnmigrationInstruction
    implements RootAction,
    StaplerProxy {
        @Override
        public String getIconFileName() {
            return null;
        }

        @Override
        public String getDisplayName() {
            return null;
        }

        @Override
        public String getUrlName() {
            return "JENKINS-24380";
        }

        public Object getTarget() {
            Jenkins.getInstance().checkPermission(Jenkins.ADMINISTER);
            return this;
        }

        public String getCommand() {
            return RunIdMigrator.getUnmigrationCommandLine(Jenkins.getInstance().getRootDir());
        }
    }
}

