/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.recovery;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Deque;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.IntSupplier;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexCommit;
import org.apache.lucene.index.IndexFormatTooNewException;
import org.apache.lucene.index.IndexFormatTooOldException;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.action.StepListener;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.action.support.ThreadedActionListener;
import org.elasticsearch.action.support.replication.ReplicationResponse;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.common.StopWatch;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.bytes.ReleasableBytesReference;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.CancellableThreads;
import org.elasticsearch.common.util.CollectionUtils;
import org.elasticsearch.common.util.concurrent.CountDown;
import org.elasticsearch.common.util.concurrent.FutureUtils;
import org.elasticsearch.common.util.concurrent.ListenableFuture;
import org.elasticsearch.core.CheckedRunnable;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Releasables;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.engine.RecoveryEngineException;
import org.elasticsearch.index.seqno.ReplicationTracker;
import org.elasticsearch.index.seqno.RetentionLease;
import org.elasticsearch.index.seqno.RetentionLeaseNotFoundException;
import org.elasticsearch.index.seqno.RetentionLeases;
import org.elasticsearch.index.seqno.SequenceNumbers;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardClosedException;
import org.elasticsearch.index.shard.IndexShardRelocatedException;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshot;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.index.store.StoreFileMetadata;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.indices.recovery.DelayRecoveryException;
import org.elasticsearch.indices.recovery.MultiChunkTransfer;
import org.elasticsearch.indices.recovery.RecoverFilesRecoveryException;
import org.elasticsearch.indices.recovery.RecoveryResponse;
import org.elasticsearch.indices.recovery.RecoveryTargetHandler;
import org.elasticsearch.indices.recovery.StartRecoveryRequest;
import org.elasticsearch.indices.recovery.plan.RecoveryPlannerService;
import org.elasticsearch.indices.recovery.plan.ShardRecoveryPlan;
import org.elasticsearch.snapshots.SnapshotShardsService;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.RemoteTransportException;
import org.elasticsearch.transport.Transports;

public class RecoverySourceHandler {
    protected final Logger logger;
    private final IndexShard shard;
    private final int shardId;
    private final StartRecoveryRequest request;
    private final int chunkSizeInBytes;
    private final RecoveryTargetHandler recoveryTarget;
    private final int maxConcurrentFileChunks;
    private final int maxConcurrentOperations;
    private final int maxConcurrentSnapshotFileDownloads;
    private final boolean useSnapshots;
    private final ThreadPool threadPool;
    private final RecoveryPlannerService recoveryPlannerService;
    private final CancellableThreads cancellableThreads = new CancellableThreads();
    private final List<Closeable> resources = new CopyOnWriteArrayList<Closeable>();
    private final ListenableFuture<RecoveryResponse> future = new ListenableFuture();

    public RecoverySourceHandler(IndexShard shard, RecoveryTargetHandler recoveryTarget, ThreadPool threadPool, StartRecoveryRequest request, int fileChunkSizeInBytes, int maxConcurrentFileChunks, int maxConcurrentOperations, int maxConcurrentSnapshotFileDownloads, boolean useSnapshots, RecoveryPlannerService recoveryPlannerService) {
        this.shard = shard;
        this.recoveryTarget = recoveryTarget;
        this.threadPool = threadPool;
        this.recoveryPlannerService = recoveryPlannerService;
        this.request = request;
        this.shardId = this.request.shardId().id();
        this.logger = Loggers.getLogger(this.getClass(), request.shardId(), "recover to " + request.targetNode().getName());
        this.chunkSizeInBytes = fileChunkSizeInBytes;
        this.maxConcurrentFileChunks = maxConcurrentFileChunks;
        this.maxConcurrentOperations = maxConcurrentOperations;
        this.maxConcurrentSnapshotFileDownloads = maxConcurrentSnapshotFileDownloads;
        this.useSnapshots = useSnapshots;
    }

    public StartRecoveryRequest getRequest() {
        return this.request;
    }

    public void addListener(ActionListener<RecoveryResponse> listener) {
        this.future.addListener(listener);
    }

    public void recoverToTarget(ActionListener<RecoveryResponse> listener) {
        this.addListener(listener);
        Closeable releaseResources = () -> IOUtils.close(this.resources);
        try {
            long startingSeqNo;
            boolean isSequenceNumberBasedRecovery;
            this.cancellableThreads.setOnCancel((reason, beforeCancelEx) -> {
                ElasticsearchException e = this.shard.state() == IndexShardState.CLOSED ? new IndexShardClosedException(this.shard.shardId(), "shard is closed and recovery was canceled reason [" + reason + "]") : new CancellableThreads.ExecutionCancelledException("recovery was canceled reason [" + reason + "]");
                if (beforeCancelEx != null) {
                    e.addSuppressed(beforeCancelEx);
                }
                IOUtils.closeWhileHandlingException(releaseResources, () -> this.future.onFailure(e));
                throw e;
            });
            Consumer<Exception> onFailure = e -> {
                assert (Transports.assertNotTransportThread(this + "[onFailure]"));
                IOUtils.closeWhileHandlingException(releaseResources, () -> this.future.onFailure((Exception)e));
            };
            SetOnce retentionLeaseRef = new SetOnce();
            RecoverySourceHandler.runUnderPrimaryPermit(() -> {
                IndexShardRoutingTable routingTable = this.shard.getReplicationGroup().getRoutingTable();
                ShardRouting targetShardRouting = routingTable.getByAllocationId(this.request.targetAllocationId());
                if (targetShardRouting == null) {
                    this.logger.debug("delaying recovery of {} as it is not listed as assigned to target node {}", (Object)this.request.shardId(), (Object)this.request.targetNode());
                    throw new DelayRecoveryException("source node does not have the shard listed in its state as allocated on the node");
                }
                assert (targetShardRouting.initializing()) : "expected recovery target to be initializing but was " + targetShardRouting;
                retentionLeaseRef.set(this.shard.getRetentionLeases().get(ReplicationTracker.getPeerRecoveryRetentionLeaseId(targetShardRouting)));
            }, this.shardId + " validating recovery target [" + this.request.targetAllocationId() + "] registered ", this.shard, this.cancellableThreads, this.logger);
            Closeable retentionLock = this.shard.acquireHistoryRetentionLock();
            this.resources.add(retentionLock);
            boolean bl = isSequenceNumberBasedRecovery = this.request.startingSeqNo() != -2L && this.isTargetSameHistory() && this.shard.hasCompleteHistoryOperations("peer-recovery", this.request.startingSeqNo()) && (retentionLeaseRef.get() == null && !this.shard.useRetentionLeasesInPeerRecovery() || retentionLeaseRef.get() != null && ((RetentionLease)retentionLeaseRef.get()).retainingSequenceNumber() <= this.request.startingSeqNo());
            if (isSequenceNumberBasedRecovery && retentionLeaseRef.get() != null) {
                retentionLock.close();
                this.logger.trace("history is retained by {}", retentionLeaseRef.get());
            } else {
                this.logger.trace("history is retained by retention lock");
            }
            StepListener<SendFileResult> sendFileStep = new StepListener<SendFileResult>();
            StepListener<TimeValue> prepareEngineStep = new StepListener<TimeValue>();
            StepListener<SendSnapshotResult> sendSnapshotStep = new StepListener<SendSnapshotResult>();
            StepListener<Void> finalizeStep = new StepListener<Void>();
            if (isSequenceNumberBasedRecovery) {
                this.logger.trace("performing sequence numbers based recovery. starting at [{}]", (Object)this.request.startingSeqNo());
                startingSeqNo = this.request.startingSeqNo();
                if (retentionLeaseRef.get() == null) {
                    this.createRetentionLease(startingSeqNo, sendFileStep.map(ignored -> SendFileResult.EMPTY));
                } else {
                    sendFileStep.onResponse(SendFileResult.EMPTY);
                }
            } else {
                Engine.IndexCommitRef safeCommitRef;
                try {
                    safeCommitRef = this.acquireSafeCommit(this.shard);
                    this.resources.add(safeCommitRef);
                }
                catch (Exception e2) {
                    throw new RecoveryEngineException(this.shard.shardId(), 1, "snapshot failed", e2);
                }
                startingSeqNo = Long.parseLong(safeCommitRef.getIndexCommit().getUserData().get("local_checkpoint")) + 1L;
                this.logger.trace("performing file-based recovery followed by history replay starting at [{}]", (Object)startingSeqNo);
                try {
                    int estimateNumOps = this.estimateNumberOfHistoryOperations(startingSeqNo);
                    Releasable releaseStore = this.acquireStore(this.shard.store());
                    this.resources.add(releaseStore);
                    sendFileStep.whenComplete(r -> IOUtils.close(safeCommitRef, releaseStore), e -> {
                        try {
                            IOUtils.close(safeCommitRef, releaseStore);
                        }
                        catch (Exception ex) {
                            this.logger.warn("releasing snapshot caused exception", (Throwable)ex);
                        }
                    });
                    StepListener<ReplicationResponse> deleteRetentionLeaseStep = new StepListener<ReplicationResponse>();
                    RecoverySourceHandler.runUnderPrimaryPermit(() -> {
                        try {
                            this.shard.removePeerRecoveryRetentionLease(this.request.targetNode().getId(), new ThreadedActionListener<ReplicationResponse>(this.logger, this.shard.getThreadPool(), "generic", deleteRetentionLeaseStep, false));
                        }
                        catch (RetentionLeaseNotFoundException e) {
                            this.logger.debug("no peer-recovery retention lease for " + this.request.targetAllocationId());
                            deleteRetentionLeaseStep.onResponse(null);
                        }
                    }, this.shardId + " removing retention lease for [" + this.request.targetAllocationId() + "]", this.shard, this.cancellableThreads, this.logger);
                    deleteRetentionLeaseStep.whenComplete(ignored -> {
                        assert (Transports.assertNotTransportThread(this + "[phase1]"));
                        this.phase1(safeCommitRef.getIndexCommit(), startingSeqNo, () -> estimateNumOps, sendFileStep);
                    }, onFailure);
                }
                catch (Exception e3) {
                    throw new RecoveryEngineException(this.shard.shardId(), 1, "sendFileStep failed", e3);
                }
            }
            assert (startingSeqNo >= 0L) : "startingSeqNo must be non negative. got: " + startingSeqNo;
            sendFileStep.whenComplete(r -> {
                assert (Transports.assertNotTransportThread(this + "[prepareTargetForTranslog]"));
                this.prepareTargetForTranslog(this.estimateNumberOfHistoryOperations(startingSeqNo), prepareEngineStep);
            }, onFailure);
            prepareEngineStep.whenComplete(prepareEngineTime -> {
                assert (Transports.assertNotTransportThread(this + "[phase2]"));
                RecoverySourceHandler.runUnderPrimaryPermit(() -> this.shard.initiateTracking(this.request.targetAllocationId()), this.shardId + " initiating tracking of " + this.request.targetAllocationId(), this.shard, this.cancellableThreads, this.logger);
                long endingSeqNo = this.shard.seqNoStats().getMaxSeqNo();
                this.logger.trace("snapshot for recovery; current size is [{}]", (Object)this.estimateNumberOfHistoryOperations(startingSeqNo));
                Translog.Snapshot phase2Snapshot = this.shard.newChangesSnapshot("peer-recovery", startingSeqNo, Long.MAX_VALUE, false, false, true);
                this.resources.add(phase2Snapshot);
                retentionLock.close();
                long maxSeenAutoIdTimestamp = this.shard.getMaxSeenAutoIdTimestamp();
                long maxSeqNoOfUpdatesOrDeletes = this.shard.getMaxSeqNoOfUpdatesOrDeletes();
                RetentionLeases retentionLeases = this.shard.getRetentionLeases();
                long mappingVersionOnPrimary = this.shard.indexSettings().getIndexMetadata().getMappingVersion();
                this.phase2(startingSeqNo, endingSeqNo, phase2Snapshot, maxSeenAutoIdTimestamp, maxSeqNoOfUpdatesOrDeletes, retentionLeases, mappingVersionOnPrimary, sendSnapshotStep);
            }, onFailure);
            long trimAboveSeqNo = startingSeqNo - 1L;
            sendSnapshotStep.whenComplete(r -> this.finalizeRecovery(r.targetLocalCheckpoint, trimAboveSeqNo, finalizeStep), onFailure);
            finalizeStep.whenComplete(r -> {
                long phase1ThrottlingWaitTime = 0L;
                SendSnapshotResult sendSnapshotResult = (SendSnapshotResult)sendSnapshotStep.result();
                SendFileResult sendFileResult = (SendFileResult)sendFileStep.result();
                RecoveryResponse response = new RecoveryResponse(sendFileResult.phase1FileNames, sendFileResult.phase1FileSizes, sendFileResult.phase1ExistingFileNames, sendFileResult.phase1ExistingFileSizes, sendFileResult.totalSize, sendFileResult.existingTotalSize, sendFileResult.took.millis(), 0L, ((TimeValue)prepareEngineStep.result()).millis(), sendSnapshotResult.sentOperations, sendSnapshotResult.tookTime.millis());
                try {
                    this.future.onResponse(response);
                }
                finally {
                    IOUtils.close(this.resources);
                }
            }, onFailure);
        }
        catch (Exception e4) {
            IOUtils.closeWhileHandlingException(releaseResources, () -> this.future.onFailure(e4));
        }
    }

    private boolean isTargetSameHistory() {
        String targetHistoryUUID = this.request.metadataSnapshot().getHistoryUUID();
        assert (targetHistoryUUID != null) : "incoming target history missing";
        return targetHistoryUUID.equals(this.shard.getHistoryUUID());
    }

    private int estimateNumberOfHistoryOperations(long startingSeqNo) throws IOException {
        return this.shard.countChanges("peer-recovery", startingSeqNo, Long.MAX_VALUE);
    }

    static void runUnderPrimaryPermit(CancellableThreads.Interruptible runnable, String reason, IndexShard primary, CancellableThreads cancellableThreads, Logger logger) {
        cancellableThreads.execute(() -> {
            final CompletableFuture permit = new CompletableFuture();
            ActionListener<Releasable> onAcquired = new ActionListener<Releasable>(){

                @Override
                public void onResponse(Releasable releasable) {
                    if (!permit.complete(releasable)) {
                        releasable.close();
                    }
                }

                @Override
                public void onFailure(Exception e) {
                    permit.completeExceptionally(e);
                }
            };
            primary.acquirePrimaryOperationPermit(onAcquired, "same", reason);
            try (Releasable ignored = (Releasable)FutureUtils.get(permit);){
                if (primary.isRelocatedPrimary()) {
                    throw new IndexShardRelocatedException(primary.shardId());
                }
                runnable.run();
            }
            finally {
                permit.whenComplete((r, e) -> {
                    if (r != null) {
                        r.close();
                    }
                    if (e != null) {
                        logger.trace("suppressing exception on completion (it was already bubbled up or the operation was aborted)", (Throwable)e);
                    }
                });
            }
        });
    }

    private Releasable acquireStore(Store store) {
        store.incRef();
        return Releasables.releaseOnce(() -> this.runWithGenericThreadPool(store::decRef));
    }

    private Engine.IndexCommitRef acquireSafeCommit(IndexShard shard) {
        Engine.IndexCommitRef commitRef = shard.acquireSafeIndexCommit();
        return new Engine.IndexCommitRef(commitRef.getIndexCommit(), () -> this.runWithGenericThreadPool(commitRef::close));
    }

    private void runWithGenericThreadPool(CheckedRunnable<Exception> task) {
        PlainActionFuture future = new PlainActionFuture();
        assert (!this.threadPool.generic().isShutdown());
        this.threadPool.generic().execute(ActionRunnable.run(future, task));
        FutureUtils.get(future);
    }

    void phase1(IndexCommit snapshot, long startingSeqNo, IntSupplier translogOps, ActionListener<SendFileResult> listener) {
        this.cancellableThreads.checkForCancel();
        Store store = this.shard.store();
        try {
            String shardStateIdentifier;
            Store.MetadataSnapshot recoverySourceMetadata;
            StopWatch stopWatch = new StopWatch().start();
            try {
                recoverySourceMetadata = store.getMetadata(snapshot);
                shardStateIdentifier = SnapshotShardsService.getShardStateId(this.shard, snapshot);
            }
            catch (CorruptIndexException | IndexFormatTooNewException | IndexFormatTooOldException ex) {
                this.shard.failShard("recovery", ex);
                throw ex;
            }
            for (String name : snapshot.getFileNames()) {
                StoreFileMetadata md = recoverySourceMetadata.get(name);
                if (md != null) continue;
                this.logger.info("Snapshot differs from actual index for file: {} meta: {}", (Object)name, (Object)recoverySourceMetadata.asMap());
                throw new CorruptIndexException("Snapshot differs from actual index - maybe index was removed metadata has " + recoverySourceMetadata.asMap().size() + " files", name);
            }
            if (!this.canSkipPhase1(recoverySourceMetadata, this.request.metadataSnapshot())) {
                this.cancellableThreads.checkForCancel();
                boolean canUseSnapshots = this.useSnapshots && this.request.canDownloadSnapshotFiles();
                this.recoveryPlannerService.computeRecoveryPlan(this.shard.shardId(), shardStateIdentifier, recoverySourceMetadata, this.request.metadataSnapshot(), startingSeqNo, translogOps.getAsInt(), this.getRequest().targetNode().getVersion(), canUseSnapshots, ActionListener.wrap(plan -> this.recoverFilesFromSourceAndSnapshot((ShardRecoveryPlan)plan, store, stopWatch, listener), listener::onFailure));
            } else {
                this.logger.trace("skipping [phase1] since source and target have identical sync id [{}]", (Object)recoverySourceMetadata.getSyncId());
                StepListener<RetentionLease> createRetentionLeaseStep = new StepListener<RetentionLease>();
                this.createRetentionLease(startingSeqNo, createRetentionLeaseStep);
                createRetentionLeaseStep.whenComplete(retentionLease -> {
                    TimeValue took = stopWatch.totalTime();
                    this.logger.trace("recovery [phase1]: took [{}]", (Object)took);
                    listener.onResponse(new SendFileResult(Collections.emptyList(), Collections.emptyList(), 0L, Collections.emptyList(), Collections.emptyList(), 0L, took));
                }, listener::onFailure);
            }
        }
        catch (Exception e) {
            throw new RecoverFilesRecoveryException(this.request.shardId(), 0, new ByteSizeValue(0L), e);
        }
    }

    void recoverFilesFromSourceAndSnapshot(final ShardRecoveryPlan shardRecoveryPlan, Store store, StopWatch stopWatch, ActionListener<SendFileResult> listener) {
        this.cancellableThreads.checkForCancel();
        List<String> filesToRecoverNames = shardRecoveryPlan.getFilesToRecoverNames();
        List<Long> filesToRecoverSizes = shardRecoveryPlan.getFilesToRecoverSizes();
        List<String> phase1ExistingFileNames = shardRecoveryPlan.getFilesPresentInTargetNames();
        List<Long> phase1ExistingFileSizes = shardRecoveryPlan.getFilesPresentInTargetSizes();
        long totalSize = shardRecoveryPlan.getTotalSize();
        long existingTotalSize = shardRecoveryPlan.getExistingSize();
        if (this.logger.isTraceEnabled()) {
            for (StoreFileMetadata md : shardRecoveryPlan.getFilesPresentInTarget()) {
                this.logger.trace("recovery [phase1]: not recovering [{}], exist in local store and has checksum [{}], size [{}]", (Object)md.name(), (Object)md.checksum(), (Object)md.length());
            }
            for (StoreFileMetadata md : shardRecoveryPlan.getSourceFilesToRecover()) {
                if (this.request.metadataSnapshot().asMap().containsKey(md.name())) {
                    this.logger.trace("recovery [phase1]: recovering [{}], exists in local store, but is different: remote [{}], local [{}]", (Object)md.name(), (Object)this.request.metadataSnapshot().asMap().get(md.name()), (Object)md);
                    continue;
                }
                this.logger.trace("recovery [phase1]: recovering [{}], does not exist in remote", (Object)md.name());
            }
            for (BlobStoreIndexShardSnapshot.FileInfo fileInfo : shardRecoveryPlan.getSnapshotFilesToRecover()) {
                StoreFileMetadata md = fileInfo.metadata();
                if (this.request.metadataSnapshot().asMap().containsKey(md.name())) {
                    this.logger.trace("recovery [phase1]: recovering [{}], exists in local store, but is different: remote [{}], local [{}]", (Object)md.name(), (Object)this.request.metadataSnapshot().asMap().get(md.name()), (Object)md);
                    continue;
                }
                this.logger.trace("recovery [phase1]: recovering [{}], does not exist in remote", (Object)md.name());
            }
            this.logger.trace("recovery [phase1]: recovering_files [{}] with total_size [{}], reusing_files [{}] with total_size [{}]", (Object)filesToRecoverNames.size(), (Object)new ByteSizeValue(totalSize), (Object)phase1ExistingFileNames.size(), (Object)new ByteSizeValue(existingTotalSize));
        }
        StepListener<Void> sendFileInfoStep = new StepListener<Void>();
        final StepListener<Tuple> recoverSnapshotFilesStep = new StepListener<Tuple>();
        StepListener<ShardRecoveryPlan> sendFilesStep = new StepListener<ShardRecoveryPlan>();
        StepListener<Tuple> createRetentionLeaseStep = new StepListener<Tuple>();
        StepListener<ShardRecoveryPlan> cleanFilesStep = new StepListener<ShardRecoveryPlan>();
        int translogOps = shardRecoveryPlan.getTranslogOps();
        this.recoveryTarget.receiveFileInfo(filesToRecoverNames, filesToRecoverSizes, phase1ExistingFileNames, phase1ExistingFileSizes, translogOps, sendFileInfoStep);
        sendFileInfoStep.whenComplete(unused -> this.recoverSnapshotFiles(shardRecoveryPlan, new ActionListener<List<StoreFileMetadata>>(){

            @Override
            public void onResponse(List<StoreFileMetadata> filesFailedToRecoverFromSnapshot) {
                recoverSnapshotFilesStep.onResponse(Tuple.tuple(shardRecoveryPlan, filesFailedToRecoverFromSnapshot));
            }

            @Override
            public void onFailure(Exception e) {
                if (!shardRecoveryPlan.canRecoverSnapshotFilesFromSourceNode() && !(e instanceof CancellableThreads.ExecutionCancelledException)) {
                    ShardRecoveryPlan fallbackPlan = shardRecoveryPlan.getFallbackPlan();
                    RecoverySourceHandler.this.recoveryTarget.receiveFileInfo(fallbackPlan.getFilesToRecoverNames(), fallbackPlan.getFilesToRecoverSizes(), fallbackPlan.getFilesPresentInTargetNames(), fallbackPlan.getFilesPresentInTargetSizes(), fallbackPlan.getTranslogOps(), recoverSnapshotFilesStep.map(r -> Tuple.tuple(fallbackPlan, Collections.emptyList())));
                } else {
                    recoverSnapshotFilesStep.onFailure(e);
                }
            }
        }), listener::onFailure);
        recoverSnapshotFilesStep.whenComplete(planAndFilesFailedToRecoverFromSnapshot -> {
            ShardRecoveryPlan recoveryPlan = (ShardRecoveryPlan)planAndFilesFailedToRecoverFromSnapshot.v1();
            List filesFailedToRecoverFromSnapshot = (List)planAndFilesFailedToRecoverFromSnapshot.v2();
            List<StoreFileMetadata> filesToRecoverFromSource = filesFailedToRecoverFromSnapshot.isEmpty() ? recoveryPlan.getSourceFilesToRecover() : CollectionUtils.concatLists(recoveryPlan.getSourceFilesToRecover(), filesFailedToRecoverFromSnapshot);
            this.sendFiles(store, filesToRecoverFromSource.toArray(new StoreFileMetadata[0]), recoveryPlan::getTranslogOps, sendFilesStep.map(unused -> recoveryPlan));
        }, listener::onFailure);
        sendFilesStep.whenComplete(recoveryPlan -> this.createRetentionLease(recoveryPlan.getStartingSeqNo(), createRetentionLeaseStep.map(retentionLease -> Tuple.tuple(recoveryPlan, retentionLease))), listener::onFailure);
        createRetentionLeaseStep.whenComplete(recoveryPlanAndRetentionLease -> {
            ShardRecoveryPlan recoveryPlan = (ShardRecoveryPlan)recoveryPlanAndRetentionLease.v1();
            RetentionLease retentionLease = (RetentionLease)recoveryPlanAndRetentionLease.v2();
            Store.MetadataSnapshot recoverySourceMetadata = recoveryPlan.getSourceMetadataSnapshot();
            long lastKnownGlobalCheckpoint = this.shard.getLastKnownGlobalCheckpoint();
            assert (retentionLease == null || retentionLease.retainingSequenceNumber() - 1L <= lastKnownGlobalCheckpoint) : retentionLease + " vs " + lastKnownGlobalCheckpoint;
            this.cleanFiles(store, recoverySourceMetadata, () -> translogOps, lastKnownGlobalCheckpoint, cleanFilesStep.map(unused -> recoveryPlan));
        }, listener::onFailure);
        cleanFilesStep.whenComplete(recoveryPlan -> {
            TimeValue took = stopWatch.totalTime();
            this.logger.trace("recovery [phase1]: took [{}]", (Object)took);
            listener.onResponse(new SendFileResult(recoveryPlan.getFilesToRecoverNames(), recoveryPlan.getFilesToRecoverSizes(), recoveryPlan.getTotalSize(), recoveryPlan.getFilesPresentInTargetNames(), recoveryPlan.getFilesPresentInTargetSizes(), recoveryPlan.getExistingSize(), took));
        }, listener::onFailure);
    }

    void recoverSnapshotFiles(ShardRecoveryPlan shardRecoveryPlan, ActionListener<List<StoreFileMetadata>> listener) {
        ShardRecoveryPlan.SnapshotFilesToRecover snapshotFilesToRecover = shardRecoveryPlan.getSnapshotFilesToRecover();
        if (snapshotFilesToRecover.isEmpty()) {
            listener.onResponse(Collections.emptyList());
            return;
        }
        new SnapshotRecoverFileRequestsSender(shardRecoveryPlan, listener).start();
    }

    void createRetentionLease(long startingSeqNo, ActionListener<RetentionLease> listener) {
        RecoverySourceHandler.runUnderPrimaryPermit(() -> {
            this.logger.trace("cloning primary's retention lease");
            try {
                StepListener<ReplicationResponse> cloneRetentionLeaseStep = new StepListener<ReplicationResponse>();
                RetentionLease clonedLease = this.shard.cloneLocalPeerRecoveryRetentionLease(this.request.targetNode().getId(), new ThreadedActionListener<ReplicationResponse>(this.logger, this.shard.getThreadPool(), "generic", cloneRetentionLeaseStep, false));
                this.logger.trace("cloned primary's retention lease as [{}]", (Object)clonedLease);
                cloneRetentionLeaseStep.addListener(listener.map(rr -> clonedLease));
            }
            catch (RetentionLeaseNotFoundException e) {
                assert (this.shard.indexSettings().getIndexVersionCreated().before(Version.V_7_4_0) || !this.shard.indexSettings().isSoftDeleteEnabled());
                StepListener<ReplicationResponse> addRetentionLeaseStep = new StepListener<ReplicationResponse>();
                long estimatedGlobalCheckpoint = startingSeqNo - 1L;
                RetentionLease newLease = this.shard.addPeerRecoveryRetentionLease(this.request.targetNode().getId(), estimatedGlobalCheckpoint, new ThreadedActionListener<ReplicationResponse>(this.logger, this.shard.getThreadPool(), "generic", addRetentionLeaseStep, false));
                addRetentionLeaseStep.addListener(listener.map(rr -> newLease));
                this.logger.trace("created retention lease with estimated checkpoint of [{}]", (Object)estimatedGlobalCheckpoint);
            }
        }, this.shardId + " establishing retention lease for [" + this.request.targetAllocationId() + "]", this.shard, this.cancellableThreads, this.logger);
    }

    boolean canSkipPhase1(Store.MetadataSnapshot source, Store.MetadataSnapshot target) {
        if (source.getSyncId() == null || !source.getSyncId().equals(target.getSyncId())) {
            return false;
        }
        if (source.getNumDocs() != target.getNumDocs()) {
            throw new IllegalStateException("try to recover " + this.request.shardId() + " from primary shard with sync id but number of docs differ: " + source.getNumDocs() + " (" + this.request.sourceNode().getName() + ", primary) vs " + target.getNumDocs() + "(" + this.request.targetNode().getName() + ")");
        }
        SequenceNumbers.CommitInfo sourceSeqNos = SequenceNumbers.loadSeqNoInfoFromLuceneCommit(source.getCommitUserData().entrySet());
        SequenceNumbers.CommitInfo targetSeqNos = SequenceNumbers.loadSeqNoInfoFromLuceneCommit(target.getCommitUserData().entrySet());
        if (sourceSeqNos.localCheckpoint != targetSeqNos.localCheckpoint || targetSeqNos.maxSeqNo != sourceSeqNos.maxSeqNo) {
            String message = "try to recover " + this.request.shardId() + " with sync id but seq_no stats are mismatched: [" + source.getCommitUserData() + "] vs [" + target.getCommitUserData() + "]";
            assert (false) : message;
            throw new IllegalStateException(message);
        }
        return true;
    }

    void prepareTargetForTranslog(int totalTranslogOps, ActionListener<TimeValue> listener) {
        StopWatch stopWatch = new StopWatch().start();
        ActionListener<Void> wrappedListener = ActionListener.wrap(nullVal -> {
            stopWatch.stop();
            TimeValue tookTime = stopWatch.totalTime();
            this.logger.trace("recovery [phase1]: remote engine start took [{}]", (Object)tookTime);
            listener.onResponse(tookTime);
        }, e -> listener.onFailure(new RecoveryEngineException(this.shard.shardId(), 1, "prepare target for translog failed", (Throwable)e)));
        this.logger.trace("recovery [phase1]: prepare remote engine for translog");
        this.cancellableThreads.checkForCancel();
        this.recoveryTarget.prepareForTranslogOperations(totalTranslogOps, wrappedListener);
    }

    void phase2(long startingSeqNo, long endingSeqNo, Translog.Snapshot snapshot, long maxSeenAutoIdTimestamp, long maxSeqNoOfUpdatesOrDeletes, RetentionLeases retentionLeases, long mappingVersion, ActionListener<SendSnapshotResult> listener) throws IOException {
        if (this.shard.state() == IndexShardState.CLOSED) {
            throw new IndexShardClosedException(this.request.shardId());
        }
        this.logger.trace("recovery [phase2]: sending transaction log operations (from [" + startingSeqNo + "] to [" + endingSeqNo + "]");
        StopWatch stopWatch = new StopWatch().start();
        StepListener<Void> sendListener = new StepListener<Void>();
        OperationBatchSender sender = new OperationBatchSender(startingSeqNo, endingSeqNo, snapshot, maxSeenAutoIdTimestamp, maxSeqNoOfUpdatesOrDeletes, retentionLeases, mappingVersion, sendListener);
        sendListener.whenComplete(ignored -> {
            long skippedOps = sender.skippedOps.get();
            int totalSentOps = sender.sentOps.get();
            long targetLocalCheckpoint = sender.targetLocalCheckpoint.get();
            assert ((long)snapshot.totalOperations() == (long)snapshot.skippedOperations() + skippedOps + (long)totalSentOps) : String.format(Locale.ROOT, "expected total [%d], overridden [%d], skipped [%d], total sent [%d]", snapshot.totalOperations(), snapshot.skippedOperations(), skippedOps, totalSentOps);
            stopWatch.stop();
            TimeValue tookTime = stopWatch.totalTime();
            this.logger.trace("recovery [phase2]: took [{}]", (Object)tookTime);
            listener.onResponse(new SendSnapshotResult(targetLocalCheckpoint, totalSentOps, tookTime));
        }, listener::onFailure);
        sender.start();
    }

    void finalizeRecovery(long targetLocalCheckpoint, long trimAboveSeqNo, ActionListener<Void> listener) {
        if (this.shard.state() == IndexShardState.CLOSED) {
            throw new IndexShardClosedException(this.request.shardId());
        }
        this.cancellableThreads.checkForCancel();
        StopWatch stopWatch = new StopWatch().start();
        this.logger.trace("finalizing recovery");
        RecoverySourceHandler.runUnderPrimaryPermit(() -> this.shard.markAllocationIdAsInSync(this.request.targetAllocationId(), targetLocalCheckpoint), this.shardId + " marking " + this.request.targetAllocationId() + " as in sync", this.shard, this.cancellableThreads, this.logger);
        long globalCheckpoint = this.shard.getLastKnownGlobalCheckpoint();
        StepListener<Void> finalizeListener = new StepListener<Void>();
        this.cancellableThreads.checkForCancel();
        this.recoveryTarget.finalizeRecovery(globalCheckpoint, trimAboveSeqNo, finalizeListener);
        finalizeListener.whenComplete(r -> {
            RecoverySourceHandler.runUnderPrimaryPermit(() -> this.shard.updateGlobalCheckpointForShard(this.request.targetAllocationId(), globalCheckpoint), this.shardId + " updating " + this.request.targetAllocationId() + "'s global checkpoint", this.shard, this.cancellableThreads, this.logger);
            if (this.request.isPrimaryRelocation()) {
                this.logger.trace("performing relocation hand-off");
                this.cancellableThreads.execute(() -> this.shard.relocated(this.request.targetAllocationId(), this.recoveryTarget::handoffPrimaryContext, ActionListener.wrap(v -> {
                    this.cancellableThreads.checkForCancel();
                    this.completeFinalizationListener(listener, stopWatch);
                }, listener::onFailure)));
            } else {
                this.completeFinalizationListener(listener, stopWatch);
            }
        }, listener::onFailure);
    }

    private void completeFinalizationListener(ActionListener<Void> listener, StopWatch stopWatch) {
        stopWatch.stop();
        this.logger.trace("finalizing recovery took [{}]", (Object)stopWatch.totalTime());
        listener.onResponse(null);
    }

    public void cancel(String reason) {
        this.cancellableThreads.cancel(reason);
        this.recoveryTarget.cancel();
    }

    public String toString() {
        return "ShardRecoveryHandler{shardId=" + this.request.shardId() + ", sourceNode=" + this.request.sourceNode() + ", targetNode=" + this.request.targetNode() + "}";
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void sendFiles(final Store store, StoreFileMetadata[] files, final IntSupplier translogOps, ActionListener<Void> listener) {
        ArrayUtil.timSort(files, Comparator.comparingLong(StoreFileMetadata::length));
        final int bufferSize = files.length == 0 ? 0 : (int)Math.min((long)this.chunkSizeInBytes, files[files.length - 1].length());
        Releasable temporaryStoreRef = this.acquireStore(store);
        try {
            final Releasable storeRef = temporaryStoreRef;
            MultiChunkTransfer<StoreFileMetadata, FileChunk> multiFileSender = new MultiChunkTransfer<StoreFileMetadata, FileChunk>(this.logger, this.threadPool.getThreadContext(), listener, this.maxConcurrentFileChunks, Arrays.asList(files)){
                final Deque<byte[]> buffers;
                final AtomicInteger liveBufferCount;
                IndexInput currentInput;
                long offset;
                {
                    super(logger, threadContext, listener, maxConcurrentChunks, sources);
                    this.buffers = new ConcurrentLinkedDeque<byte[]>();
                    this.liveBufferCount = new AtomicInteger();
                    this.currentInput = null;
                    this.offset = 0L;
                }

                @Override
                protected void onNewResource(StoreFileMetadata md) throws IOException {
                    this.offset = 0L;
                    IOUtils.close((Closeable)this.currentInput);
                    this.currentInput = md.hashEqualsContents() ? null : store.directory().openInput(md.name(), IOContext.READONCE);
                }

                @Override
                protected FileChunk nextChunkRequest(StoreFileMetadata md) throws IOException {
                    assert (Transports.assertNotTransportThread("read file chunk"));
                    RecoverySourceHandler.this.cancellableThreads.checkForCancel();
                    if (this.currentInput == null) {
                        assert (md.hashEqualsContents());
                        return new FileChunk(md, new BytesArray(md.hash()), 0L, true, () -> {});
                    }
                    byte[] buffer = Objects.requireNonNullElseGet(this.buffers.pollFirst(), () -> new byte[bufferSize]);
                    assert (this.liveBufferCount.incrementAndGet() > 0);
                    int toRead = Math.toIntExact(Math.min(md.length() - this.offset, (long)buffer.length));
                    this.currentInput.readBytes(buffer, 0, toRead, false);
                    boolean lastChunk = this.offset + (long)toRead == md.length();
                    FileChunk chunk = new FileChunk(md, new BytesArray(buffer, 0, toRead), this.offset, lastChunk, () -> {
                        assert (this.liveBufferCount.decrementAndGet() >= 0);
                        this.buffers.addFirst(buffer);
                    });
                    this.offset += (long)toRead;
                    return chunk;
                }

                @Override
                protected void executeChunkRequest(FileChunk request, ActionListener<Void> listener) {
                    RecoverySourceHandler.this.cancellableThreads.checkForCancel();
                    ReleasableBytesReference content = new ReleasableBytesReference(request.content, request);
                    RecoverySourceHandler.this.recoveryTarget.writeFileChunk(request.md, request.position, content, request.lastChunk, translogOps.getAsInt(), ActionListener.runBefore(listener, content::close));
                }

                @Override
                protected void handleError(StoreFileMetadata md, Exception e) throws Exception {
                    RecoverySourceHandler.this.handleErrorOnSendFiles(store, e, new StoreFileMetadata[]{md});
                }

                @Override
                public void close() throws IOException {
                    IOUtils.close(this.currentInput, storeRef);
                }

                @Override
                protected boolean assertOnSuccess() {
                    assert (this.liveBufferCount.get() == 0) : "leaked [" + this.liveBufferCount + "] buffers";
                    return true;
                }
            };
            this.resources.add(multiFileSender);
            temporaryStoreRef = null;
            multiFileSender.start();
        }
        finally {
            Releasables.close(temporaryStoreRef);
        }
    }

    private void cleanFiles(Store store, Store.MetadataSnapshot sourceMetadata, IntSupplier translogOps, long globalCheckpoint, ActionListener<Void> listener) {
        this.cancellableThreads.checkForCancel();
        this.recoveryTarget.cleanFiles(translogOps.getAsInt(), globalCheckpoint, sourceMetadata, listener.delegateResponse((l, e) -> ActionListener.completeWith(l, () -> {
            StoreFileMetadata[] mds = (StoreFileMetadata[])StreamSupport.stream(sourceMetadata.spliterator(), false).toArray(StoreFileMetadata[]::new);
            ArrayUtil.timSort(mds, Comparator.comparingLong(StoreFileMetadata::length));
            this.handleErrorOnSendFiles(store, (Exception)e, mds);
            throw e;
        })));
    }

    private void handleErrorOnSendFiles(Store store, Exception e, StoreFileMetadata[] mds) throws Exception {
        IOException corruptIndexException = ExceptionsHelper.unwrapCorruption(e);
        assert (Transports.assertNotTransportThread(this + "[handle error on send/clean files]"));
        if (corruptIndexException != null) {
            IOException localException = null;
            for (StoreFileMetadata md : mds) {
                this.cancellableThreads.checkForCancel();
                this.logger.debug("checking integrity for file {} after remove corruption exception", (Object)md);
                if (store.checkIntegrityNoException(md)) continue;
                this.logger.warn("{} Corrupted file detected {} checksum mismatch", (Object)this.shardId, (Object)md);
                if (localException == null) {
                    localException = corruptIndexException;
                }
                this.failEngine(corruptIndexException);
            }
            if (localException != null) {
                throw localException;
            }
            RemoteTransportException remoteException = new RemoteTransportException("File corruption occurred on recovery but checksums are ok", null);
            remoteException.addSuppressed(e);
            this.logger.warn(() -> new ParameterizedMessage("{} Remote file corruption on node {}, recovering {}. local checksum OK", this.shardId, this.request.targetNode(), mds), (Throwable)corruptIndexException);
            throw remoteException;
        }
        throw e;
    }

    protected void failEngine(IOException cause) {
        this.shard.failShard("recovery", cause);
    }

    static final class SendFileResult {
        final List<String> phase1FileNames;
        final List<Long> phase1FileSizes;
        final long totalSize;
        final List<String> phase1ExistingFileNames;
        final List<Long> phase1ExistingFileSizes;
        final long existingTotalSize;
        final TimeValue took;
        static final SendFileResult EMPTY = new SendFileResult(Collections.emptyList(), Collections.emptyList(), 0L, Collections.emptyList(), Collections.emptyList(), 0L, TimeValue.ZERO);

        SendFileResult(List<String> phase1FileNames, List<Long> phase1FileSizes, long totalSize, List<String> phase1ExistingFileNames, List<Long> phase1ExistingFileSizes, long existingTotalSize, TimeValue took) {
            this.phase1FileNames = phase1FileNames;
            this.phase1FileSizes = phase1FileSizes;
            this.totalSize = totalSize;
            this.phase1ExistingFileNames = phase1ExistingFileNames;
            this.phase1ExistingFileSizes = phase1ExistingFileSizes;
            this.existingTotalSize = existingTotalSize;
            this.took = took;
        }
    }

    private class SnapshotRecoverFileRequestsSender {
        private final ShardRecoveryPlan shardRecoveryPlan;
        private final ShardRecoveryPlan.SnapshotFilesToRecover snapshotFilesToRecover;
        private final ActionListener<List<StoreFileMetadata>> listener;
        private final CountDown countDown;
        private final BlockingQueue<BlobStoreIndexShardSnapshot.FileInfo> pendingSnapshotFilesToRecover;
        private final AtomicBoolean cancelled = new AtomicBoolean();
        private final Set<ListenableFuture<Void>> outstandingRequests;
        private List<StoreFileMetadata> filesFailedToDownloadFromSnapshot;

        SnapshotRecoverFileRequestsSender(ShardRecoveryPlan shardRecoveryPlan, ActionListener<List<StoreFileMetadata>> listener) {
            this.outstandingRequests = new HashSet<ListenableFuture<Void>>(RecoverySourceHandler.this.maxConcurrentSnapshotFileDownloads);
            this.shardRecoveryPlan = shardRecoveryPlan;
            this.snapshotFilesToRecover = shardRecoveryPlan.getSnapshotFilesToRecover();
            this.listener = listener;
            this.countDown = new CountDown(shardRecoveryPlan.getSnapshotFilesToRecover().size());
            this.pendingSnapshotFilesToRecover = new LinkedBlockingQueue<BlobStoreIndexShardSnapshot.FileInfo>(shardRecoveryPlan.getSnapshotFilesToRecover().getSnapshotFiles());
        }

        void start() {
            for (int i = 0; i < RecoverySourceHandler.this.maxConcurrentSnapshotFileDownloads; ++i) {
                this.sendRequest();
            }
        }

        void sendRequest() {
            final BlobStoreIndexShardSnapshot.FileInfo snapshotFileToRecover = (BlobStoreIndexShardSnapshot.FileInfo)this.pendingSnapshotFilesToRecover.poll();
            if (snapshotFileToRecover == null) {
                return;
            }
            ListenableFuture<Void> requestFuture = new ListenableFuture<Void>();
            try {
                RecoverySourceHandler.this.cancellableThreads.checkForCancel();
                ActionListener<Void> sendRequestListener = new ActionListener<Void>(){

                    @Override
                    public void onResponse(Void unused) {
                        SnapshotRecoverFileRequestsSender.this.onRequestCompletion(snapshotFileToRecover.metadata(), null);
                    }

                    @Override
                    public void onFailure(Exception e) {
                        RecoverySourceHandler.this.logger.warn(new ParameterizedMessage("failed to recover file [{}] from snapshot, will recover from primary instead", (Object)snapshotFileToRecover.metadata()), (Throwable)e);
                        if (SnapshotRecoverFileRequestsSender.this.shardRecoveryPlan.canRecoverSnapshotFilesFromSourceNode()) {
                            SnapshotRecoverFileRequestsSender.this.onRequestCompletion(snapshotFileToRecover.metadata(), e);
                        } else {
                            SnapshotRecoverFileRequestsSender.this.cancel(e);
                        }
                    }
                };
                requestFuture.addListener(sendRequestListener);
                this.trackOutstandingRequest(requestFuture);
                RecoverySourceHandler.this.recoveryTarget.restoreFileFromSnapshot(this.snapshotFilesToRecover.getRepository(), this.snapshotFilesToRecover.getIndexId(), snapshotFileToRecover, ActionListener.runBefore(requestFuture, () -> this.unTrackOutstandingRequest(requestFuture)));
            }
            catch (CancellableThreads.ExecutionCancelledException e) {
                this.cancel(e);
            }
            catch (Exception e) {
                this.unTrackOutstandingRequest(requestFuture);
                this.onRequestCompletion(snapshotFileToRecover.metadata(), e);
            }
        }

        void cancel(Exception e) {
            if (this.cancelled.compareAndSet(false, true)) {
                this.pendingSnapshotFilesToRecover.clear();
                this.notifyFailureOnceAllOutstandingRequestAreDone(e);
            }
        }

        void onRequestCompletion(StoreFileMetadata storeFileMetadata, @Nullable Exception exception) {
            if (this.cancelled.get()) {
                return;
            }
            if (exception != null) {
                this.addFileFailedToRecoverFromSnapshot(storeFileMetadata);
            }
            if (this.countDown.countDown()) {
                List<StoreFileMetadata> failedToRecoverFromSnapshotFiles = this.getFilesFailedToRecoverFromSnapshot();
                this.listener.onResponse(failedToRecoverFromSnapshotFiles);
            } else {
                this.sendRequest();
            }
        }

        synchronized void addFileFailedToRecoverFromSnapshot(StoreFileMetadata storeFileMetadata) {
            if (this.filesFailedToDownloadFromSnapshot == null) {
                this.filesFailedToDownloadFromSnapshot = new ArrayList<StoreFileMetadata>();
            }
            this.filesFailedToDownloadFromSnapshot.add(storeFileMetadata);
        }

        synchronized List<StoreFileMetadata> getFilesFailedToRecoverFromSnapshot() {
            return Objects.requireNonNullElse(this.filesFailedToDownloadFromSnapshot, Collections.emptyList());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void trackOutstandingRequest(ListenableFuture<Void> future) {
            boolean cancelled;
            Set<ListenableFuture<Void>> set = this.outstandingRequests;
            synchronized (set) {
                boolean bl = cancelled = RecoverySourceHandler.this.cancellableThreads.isCancelled() || this.cancelled.get();
                if (!cancelled) {
                    this.outstandingRequests.add(future);
                }
            }
            if (cancelled) {
                RecoverySourceHandler.this.cancellableThreads.checkForCancel();
                assert (this.cancelled.get());
                throw new CancellableThreads.ExecutionCancelledException("Recover snapshot files cancelled");
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void unTrackOutstandingRequest(ListenableFuture<Void> future) {
            Set<ListenableFuture<Void>> set = this.outstandingRequests;
            synchronized (set) {
                this.outstandingRequests.remove(future);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void notifyFailureOnceAllOutstandingRequestAreDone(Exception e) {
            HashSet<ListenableFuture<Void>> pendingRequests;
            assert (this.cancelled.get());
            Set<ListenableFuture<Void>> set = this.outstandingRequests;
            synchronized (set) {
                pendingRequests = new HashSet<ListenableFuture<Void>>(this.outstandingRequests);
            }
            if (pendingRequests.isEmpty()) {
                this.listener.onFailure(e);
                return;
            }
            CountDown pendingRequestsCountDown = new CountDown(pendingRequests.size());
            for (ListenableFuture listenableFuture : pendingRequests) {
                listenableFuture.addListener(ActionListener.wrap(() -> {
                    if (pendingRequestsCountDown.countDown()) {
                        this.listener.onFailure(e);
                    }
                }));
            }
        }
    }

    private class OperationBatchSender
    extends MultiChunkTransfer<Translog.Snapshot, OperationChunkRequest> {
        private final long startingSeqNo;
        private final long endingSeqNo;
        private final Translog.Snapshot snapshot;
        private final long maxSeenAutoIdTimestamp;
        private final long maxSeqNoOfUpdatesOrDeletes;
        private final RetentionLeases retentionLeases;
        private final long mappingVersion;
        private int lastBatchCount;
        private final AtomicInteger skippedOps;
        private final AtomicInteger sentOps;
        private final AtomicLong targetLocalCheckpoint;

        OperationBatchSender(long startingSeqNo, long endingSeqNo, Translog.Snapshot snapshot, long maxSeenAutoIdTimestamp, long maxSeqNoOfUpdatesOrDeletes, RetentionLeases retentionLeases, long mappingVersion, ActionListener<Void> listener) {
            super(RecoverySourceHandler.this.logger, RecoverySourceHandler.this.threadPool.getThreadContext(), listener, RecoverySourceHandler.this.maxConcurrentOperations, List.of(snapshot));
            this.lastBatchCount = 0;
            this.skippedOps = new AtomicInteger();
            this.sentOps = new AtomicInteger();
            this.targetLocalCheckpoint = new AtomicLong(-1L);
            this.startingSeqNo = startingSeqNo;
            this.endingSeqNo = endingSeqNo;
            this.snapshot = snapshot;
            this.maxSeenAutoIdTimestamp = maxSeenAutoIdTimestamp;
            this.maxSeqNoOfUpdatesOrDeletes = maxSeqNoOfUpdatesOrDeletes;
            this.retentionLeases = retentionLeases;
            this.mappingVersion = mappingVersion;
        }

        @Override
        protected synchronized OperationChunkRequest nextChunkRequest(Translog.Snapshot snapshot) throws IOException {
            Translog.Operation operation;
            assert (Transports.assertNotTransportThread("[phase2]"));
            RecoverySourceHandler.this.cancellableThreads.checkForCancel();
            ArrayList<Translog.Operation> ops = this.lastBatchCount > 0 ? new ArrayList<Translog.Operation>(this.lastBatchCount) : new ArrayList();
            long batchSizeInBytes = 0L;
            while ((operation = snapshot.next()) != null) {
                if (RecoverySourceHandler.this.shard.state() == IndexShardState.CLOSED) {
                    throw new IndexShardClosedException(RecoverySourceHandler.this.request.shardId());
                }
                long seqNo = operation.seqNo();
                if (seqNo < this.startingSeqNo || seqNo > this.endingSeqNo) {
                    this.skippedOps.incrementAndGet();
                    continue;
                }
                ops.add(operation);
                this.sentOps.incrementAndGet();
                if ((batchSizeInBytes += operation.estimateSize()) < (long)RecoverySourceHandler.this.chunkSizeInBytes) continue;
                break;
            }
            this.lastBatchCount = ops.size();
            return new OperationChunkRequest(ops, operation == null);
        }

        @Override
        protected void executeChunkRequest(OperationChunkRequest request, ActionListener<Void> listener) {
            RecoverySourceHandler.this.cancellableThreads.checkForCancel();
            RecoverySourceHandler.this.recoveryTarget.indexTranslogOperations(request.operations, this.snapshot.totalOperations(), this.maxSeenAutoIdTimestamp, this.maxSeqNoOfUpdatesOrDeletes, this.retentionLeases, this.mappingVersion, listener.delegateFailure((l, newCheckpoint) -> {
                this.targetLocalCheckpoint.updateAndGet(curr -> SequenceNumbers.max(curr, newCheckpoint));
                l.onResponse(null);
            }));
        }

        @Override
        protected void handleError(Translog.Snapshot snapshot, Exception e) {
            throw new RecoveryEngineException(RecoverySourceHandler.this.shard.shardId(), 2, "failed to send/replay operations", e);
        }

        @Override
        public void close() throws IOException {
            this.snapshot.close();
        }
    }

    static final class SendSnapshotResult {
        final long targetLocalCheckpoint;
        final int sentOperations;
        final TimeValue tookTime;

        SendSnapshotResult(long targetLocalCheckpoint, int sentOperations, TimeValue tookTime) {
            this.targetLocalCheckpoint = targetLocalCheckpoint;
            this.sentOperations = sentOperations;
            this.tookTime = tookTime;
        }
    }

    private static class FileChunk
    implements MultiChunkTransfer.ChunkRequest,
    Releasable {
        final StoreFileMetadata md;
        final BytesReference content;
        final long position;
        final boolean lastChunk;
        final Releasable onClose;

        FileChunk(StoreFileMetadata md, BytesReference content, long position, boolean lastChunk, Releasable onClose) {
            this.md = md;
            this.content = content;
            this.position = position;
            this.lastChunk = lastChunk;
            this.onClose = onClose;
        }

        @Override
        public boolean lastChunk() {
            return this.lastChunk;
        }

        @Override
        public void close() {
            this.onClose.close();
        }
    }

    private static class OperationChunkRequest
    implements MultiChunkTransfer.ChunkRequest {
        final List<Translog.Operation> operations;
        final boolean lastChunk;

        OperationChunkRequest(List<Translog.Operation> operations, boolean lastChunk) {
            this.operations = operations;
            this.lastChunk = lastChunk;
        }

        @Override
        public boolean lastChunk() {
            return this.lastChunk;
        }
    }
}

