/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.gateway;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.block.ClusterBlock;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.allocation.AllocationService;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.gateway.ClusterStateUpdaters;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.threadpool.ThreadPool;

public class GatewayService
extends AbstractLifecycleComponent
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(GatewayService.class);
    public static final Setting<Integer> EXPECTED_DATA_NODES_SETTING = Setting.intSetting("gateway.expected_data_nodes", -1, -1, Setting.Property.NodeScope);
    public static final Setting<TimeValue> RECOVER_AFTER_TIME_SETTING = Setting.positiveTimeSetting("gateway.recover_after_time", TimeValue.timeValueMillis(0L), Setting.Property.NodeScope);
    public static final Setting<Integer> RECOVER_AFTER_DATA_NODES_SETTING = Setting.intSetting("gateway.recover_after_data_nodes", -1, -1, Setting.Property.NodeScope);
    public static final ClusterBlock STATE_NOT_RECOVERED_BLOCK = new ClusterBlock(1, "state not recovered / initialized", true, true, false, RestStatus.SERVICE_UNAVAILABLE, ClusterBlockLevel.ALL);
    static final TimeValue DEFAULT_RECOVER_AFTER_TIME_IF_EXPECTED_NODES_IS_SET = TimeValue.timeValueMinutes(5L);
    private final ThreadPool threadPool;
    private final AllocationService allocationService;
    private final ClusterService clusterService;
    private final TimeValue recoverAfterTime;
    private final int recoverAfterDataNodes;
    private final int expectedDataNodes;
    private final AtomicBoolean recoveryInProgress = new AtomicBoolean();
    private final AtomicBoolean scheduledRecovery = new AtomicBoolean();

    @Inject
    public GatewayService(Settings settings, AllocationService allocationService, ClusterService clusterService, ThreadPool threadPool) {
        this.allocationService = allocationService;
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        this.expectedDataNodes = EXPECTED_DATA_NODES_SETTING.get(settings);
        this.recoverAfterTime = RECOVER_AFTER_TIME_SETTING.exists(settings) ? RECOVER_AFTER_TIME_SETTING.get(settings) : (this.expectedDataNodes >= 0 ? DEFAULT_RECOVER_AFTER_TIME_IF_EXPECTED_NODES_IS_SET : null);
        this.recoverAfterDataNodes = RECOVER_AFTER_DATA_NODES_SETTING.get(settings);
    }

    @Override
    protected void doStart() {
        if (DiscoveryNode.isMasterNode(this.clusterService.getSettings())) {
            this.clusterService.addListener(this);
        }
    }

    @Override
    protected void doStop() {
        this.clusterService.removeListener(this);
    }

    @Override
    protected void doClose() {
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (this.lifecycle.stoppedOrClosed()) {
            return;
        }
        ClusterState state = event.state();
        if (!state.nodes().isLocalNodeElectedMaster()) {
            return;
        }
        if (!state.blocks().hasGlobalBlock(STATE_NOT_RECOVERED_BLOCK)) {
            return;
        }
        DiscoveryNodes nodes = state.nodes();
        if (state.nodes().getMasterNodeId() == null) {
            logger.debug("not recovering from gateway, no master elected yet");
        } else if (this.recoverAfterDataNodes != -1 && nodes.getDataNodes().size() < this.recoverAfterDataNodes) {
            logger.debug("not recovering from gateway, nodes_size (data) [{}] < recover_after_data_nodes [{}]", (Object)nodes.getDataNodes().size(), (Object)this.recoverAfterDataNodes);
        } else {
            Object reason;
            boolean enforceRecoverAfterTime;
            if (this.expectedDataNodes == -1) {
                enforceRecoverAfterTime = true;
                reason = "recover_after_time was set to [" + this.recoverAfterTime + "]";
            } else if (this.expectedDataNodes <= nodes.getDataNodes().size()) {
                enforceRecoverAfterTime = false;
                reason = "";
            } else {
                enforceRecoverAfterTime = true;
                reason = "expecting [" + this.expectedDataNodes + "] data nodes, but only have [" + nodes.getDataNodes().size() + "]";
            }
            this.performStateRecovery(enforceRecoverAfterTime, (String)reason);
        }
    }

    private void performStateRecovery(boolean enforceRecoverAfterTime, String reason) {
        if (enforceRecoverAfterTime && this.recoverAfterTime != null) {
            if (this.scheduledRecovery.compareAndSet(false, true)) {
                logger.info("delaying initial state recovery for [{}]. {}", (Object)this.recoverAfterTime, (Object)reason);
                this.threadPool.schedule(new AbstractRunnable(){

                    @Override
                    public void onFailure(Exception e) {
                        logger.warn("delayed state recovery failed", (Throwable)e);
                        GatewayService.this.resetRecoveredFlags();
                    }

                    @Override
                    protected void doRun() {
                        if (GatewayService.this.recoveryInProgress.compareAndSet(false, true)) {
                            logger.info("recover_after_time [{}] elapsed. performing state recovery...", (Object)GatewayService.this.recoverAfterTime);
                            GatewayService.this.runRecovery();
                        }
                    }
                }, this.recoverAfterTime, "generic");
            }
        } else if (this.recoveryInProgress.compareAndSet(false, true)) {
            this.threadPool.generic().execute(new AbstractRunnable(){

                @Override
                public void onFailure(Exception e) {
                    logger.warn("state recovery failed", (Throwable)e);
                    GatewayService.this.resetRecoveredFlags();
                }

                @Override
                protected void doRun() {
                    logger.debug("performing state recovery...");
                    GatewayService.this.runRecovery();
                }
            });
        }
    }

    private void resetRecoveredFlags() {
        this.recoveryInProgress.set(false);
        this.scheduledRecovery.set(false);
    }

    TimeValue recoverAfterTime() {
        return this.recoverAfterTime;
    }

    private void runRecovery() {
        this.clusterService.submitStateUpdateTask("local-gateway-elected-state", new RecoverStateUpdateTask());
    }

    class RecoverStateUpdateTask
    extends ClusterStateUpdateTask {
        RecoverStateUpdateTask() {
        }

        @Override
        public ClusterState execute(ClusterState currentState) {
            if (!currentState.blocks().hasGlobalBlock(STATE_NOT_RECOVERED_BLOCK)) {
                logger.debug("cluster is already recovered");
                return currentState;
            }
            ClusterState newState = Function.identity().andThen(ClusterStateUpdaters::updateRoutingTable).andThen(ClusterStateUpdaters::removeStateNotRecoveredBlock).apply(currentState);
            return GatewayService.this.allocationService.reroute(newState, "state recovered");
        }

        @Override
        public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
            logger.info("recovered [{}] indices into cluster_state", (Object)newState.metadata().indices().size());
            GatewayService.this.resetRecoveredFlags();
        }

        @Override
        public void onNoLongerMaster(String source) {
            logger.debug("stepped down as master before recovering state [{}]", (Object)source);
            GatewayService.this.resetRecoveredFlags();
        }

        @Override
        public void onFailure(String source, Exception e) {
            logger.info(() -> new ParameterizedMessage("unexpected failure during [{}]", (Object)source), (Throwable)e);
            GatewayService.this.resetRecoveredFlags();
        }
    }
}

