/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.coordination;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateTaskExecutor;
import org.elasticsearch.cluster.NotMasterException;
import org.elasticsearch.cluster.block.ClusterBlocks;
import org.elasticsearch.cluster.coordination.CoordinationMetadata;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.RerouteService;
import org.elasticsearch.cluster.routing.allocation.AllocationService;
import org.elasticsearch.common.Priority;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;

public class JoinTaskExecutor
implements ClusterStateTaskExecutor<Task> {
    private final AllocationService allocationService;
    private final Logger logger;
    private final RerouteService rerouteService;

    public JoinTaskExecutor(AllocationService allocationService, Logger logger, RerouteService rerouteService) {
        this.allocationService = allocationService;
        this.logger = logger;
        this.rerouteService = rerouteService;
    }

    @Override
    public ClusterStateTaskExecutor.ClusterTasksResult<Task> execute(ClusterState currentState, List<Task> joiningNodes) throws Exception {
        ClusterState.Builder newState;
        ClusterStateTaskExecutor.ClusterTasksResult.Builder<Task> results = ClusterStateTaskExecutor.ClusterTasksResult.builder();
        DiscoveryNodes currentNodes = currentState.nodes();
        boolean nodesChanged = false;
        if (joiningNodes.size() == 1 && joiningNodes.get(0).isFinishElectionTask()) {
            return results.successes(joiningNodes).build(currentState);
        }
        if (currentNodes.getMasterNode() == null && joiningNodes.stream().anyMatch(Task::isBecomeMasterTask)) {
            assert (joiningNodes.stream().anyMatch(Task::isFinishElectionTask)) : "becoming a master but election is not finished " + joiningNodes;
            newState = this.becomeMasterAndTrimConflictingNodes(currentState, joiningNodes);
            nodesChanged = true;
        } else {
            if (!currentNodes.isLocalNodeElectedMaster()) {
                this.logger.trace("processing node joins, but we are not the master. current master: {}", (Object)currentNodes.getMasterNode());
                throw new NotMasterException("Node [" + currentNodes.getLocalNode() + "] not master for join request");
            }
            newState = ClusterState.builder(currentState);
        }
        DiscoveryNodes.Builder nodesBuilder = DiscoveryNodes.builder(newState.nodes());
        assert (nodesBuilder.isLocalNodeElectedMaster());
        Version minClusterNodeVersion = newState.nodes().getMinNodeVersion();
        Version maxClusterNodeVersion = newState.nodes().getMaxNodeVersion();
        boolean enforceVersionBarrier = !currentState.getBlocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK);
        HashMap<String, String> joiniedNodeNameIds = new HashMap<String, String>();
        for (Task joinTask : joiningNodes) {
            block16: {
                if (!joinTask.isBecomeMasterTask() && !joinTask.isFinishElectionTask()) {
                    if (currentNodes.nodeExistsWithSameRoles(joinTask.node())) {
                        this.logger.debug("received a join request for an existing node [{}]", (Object)joinTask.node());
                    } else {
                        DiscoveryNode node = joinTask.node();
                        try {
                            if (enforceVersionBarrier) {
                                JoinTaskExecutor.ensureVersionBarrier(node.getVersion(), minClusterNodeVersion);
                            }
                            JoinTaskExecutor.ensureNodesCompatibility(node.getVersion(), minClusterNodeVersion, maxClusterNodeVersion);
                            JoinTaskExecutor.ensureIndexCompatibility(node.getVersion(), currentState.getMetadata());
                            nodesBuilder.add(node);
                            nodesChanged = true;
                            minClusterNodeVersion = Version.min(minClusterNodeVersion, node.getVersion());
                            maxClusterNodeVersion = Version.max(maxClusterNodeVersion, node.getVersion());
                            if (!node.isMasterNode()) break block16;
                            joiniedNodeNameIds.put(node.getName(), node.getId());
                        }
                        catch (IllegalArgumentException | IllegalStateException e2) {
                            results.failure(joinTask, e2);
                            continue;
                        }
                    }
                }
            }
            results.success(joinTask);
        }
        if (nodesChanged) {
            Set<CoordinationMetadata.VotingConfigExclusion> currentVotingConfigExclusions;
            Set<CoordinationMetadata.VotingConfigExclusion> newVotingConfigExclusions;
            this.rerouteService.reroute("post-join reroute", Priority.HIGH, ActionListener.wrap(r -> this.logger.trace("post-join reroute completed"), e -> this.logger.debug("post-join reroute failed", (Throwable)e)));
            if (!joiniedNodeNameIds.isEmpty() && !(newVotingConfigExclusions = (currentVotingConfigExclusions = currentState.getVotingConfigExclusions()).stream().map(e -> {
                if ("_absent_".equals(e.getNodeId()) && joiniedNodeNameIds.containsKey(e.getNodeName())) {
                    return new CoordinationMetadata.VotingConfigExclusion((String)joiniedNodeNameIds.get(e.getNodeName()), e.getNodeName());
                }
                return e;
            }).collect(Collectors.toSet())).equals(currentVotingConfigExclusions)) {
                CoordinationMetadata.Builder coordMetadataBuilder = CoordinationMetadata.builder(currentState.coordinationMetadata()).clearVotingConfigExclusions();
                newVotingConfigExclusions.forEach(coordMetadataBuilder::addVotingConfigExclusion);
                Metadata newMetadata = Metadata.builder(currentState.metadata()).coordinationMetadata(coordMetadataBuilder.build()).build();
                return results.build(this.allocationService.adaptAutoExpandReplicas(newState.nodes(nodesBuilder).metadata(newMetadata).build()));
            }
            ClusterState updatedState = this.allocationService.adaptAutoExpandReplicas(newState.nodes(nodesBuilder).build());
            assert (!enforceVersionBarrier || updatedState.nodes().getMinNodeVersion().onOrAfter(currentState.nodes().getMinNodeVersion())) : "min node version decreased from [" + currentState.nodes().getMinNodeVersion() + "] to [" + updatedState.nodes().getMinNodeVersion() + "]";
            return results.build(updatedState);
        }
        return results.build(newState.build());
    }

    protected ClusterState.Builder becomeMasterAndTrimConflictingNodes(ClusterState currentState, List<Task> joiningNodes) {
        assert (currentState.nodes().getMasterNodeId() == null) : currentState;
        DiscoveryNodes currentNodes = currentState.nodes();
        DiscoveryNodes.Builder nodesBuilder = DiscoveryNodes.builder(currentNodes);
        nodesBuilder.masterNodeId(currentState.nodes().getLocalNodeId());
        for (Task joinTask : joiningNodes) {
            DiscoveryNode nodeWithSameAddress;
            if (joinTask.isBecomeMasterTask() || joinTask.isFinishElectionTask()) continue;
            DiscoveryNode joiningNode = joinTask.node();
            DiscoveryNode nodeWithSameId = nodesBuilder.get(joiningNode.getId());
            if (nodeWithSameId != null && !nodeWithSameId.equals(joiningNode)) {
                this.logger.debug("removing existing node [{}], which conflicts with incoming join from [{}]", (Object)nodeWithSameId, (Object)joiningNode);
                nodesBuilder.remove(nodeWithSameId.getId());
            }
            if ((nodeWithSameAddress = currentNodes.findByAddress(joiningNode.getAddress())) == null || nodeWithSameAddress.equals(joiningNode)) continue;
            this.logger.debug("removing existing node [{}], which conflicts with incoming join from [{}]", (Object)nodeWithSameAddress, (Object)joiningNode);
            nodesBuilder.remove(nodeWithSameAddress.getId());
        }
        ClusterState tmpState = ClusterState.builder(currentState).nodes(nodesBuilder).blocks(ClusterBlocks.builder().blocks(currentState.blocks()).removeGlobalBlock(2)).build();
        this.logger.trace("becomeMasterAndTrimConflictingNodes: {}", (Object)tmpState.nodes());
        this.allocationService.cleanCaches();
        tmpState = PersistentTasksCustomMetadata.disassociateDeadNodes(tmpState);
        return ClusterState.builder(this.allocationService.disassociateDeadNodes(tmpState, false, "removed dead nodes on election"));
    }

    @Override
    public boolean runOnlyOnMaster() {
        return false;
    }

    public static Task newBecomeMasterTask() {
        return new Task(null, "_BECOME_MASTER_TASK_");
    }

    public static Task newFinishElectionTask() {
        return new Task(null, "_FINISH_ELECTION_");
    }

    public static void ensureIndexCompatibility(Version nodeVersion, Metadata metadata) {
        Version supportedIndexVersion = nodeVersion.minimumIndexCompatibilityVersion();
        for (IndexMetadata idxMetadata : metadata) {
            if (idxMetadata.getCreationVersion().after(nodeVersion)) {
                throw new IllegalStateException("index " + idxMetadata.getIndex() + " version not supported: " + idxMetadata.getCreationVersion() + " the node version is: " + nodeVersion);
            }
            if (!idxMetadata.getCreationVersion().before(supportedIndexVersion)) continue;
            throw new IllegalStateException("index " + idxMetadata.getIndex() + " version not supported: " + idxMetadata.getCreationVersion() + " minimum compatible index version is: " + supportedIndexVersion);
        }
    }

    public static void ensureNodesCompatibility(Version joiningNodeVersion, DiscoveryNodes currentNodes) {
        Version minNodeVersion = currentNodes.getMinNodeVersion();
        Version maxNodeVersion = currentNodes.getMaxNodeVersion();
        JoinTaskExecutor.ensureNodesCompatibility(joiningNodeVersion, minNodeVersion, maxNodeVersion);
    }

    public static void ensureNodesCompatibility(Version joiningNodeVersion, Version minClusterNodeVersion, Version maxClusterNodeVersion) {
        assert (minClusterNodeVersion.onOrBefore(maxClusterNodeVersion)) : minClusterNodeVersion + " > " + maxClusterNodeVersion;
        if (!joiningNodeVersion.isCompatible(maxClusterNodeVersion)) {
            throw new IllegalStateException("node version [" + joiningNodeVersion + "] is not supported. The cluster contains nodes with version [" + maxClusterNodeVersion + "], which is incompatible.");
        }
        if (!joiningNodeVersion.isCompatible(minClusterNodeVersion)) {
            throw new IllegalStateException("node version [" + joiningNodeVersion + "] is not supported.The cluster contains nodes with version [" + minClusterNodeVersion + "], which is incompatible.");
        }
    }

    public static void ensureVersionBarrier(Version joiningNodeVersion, Version minClusterNodeVersion) {
        if (joiningNodeVersion.before(minClusterNodeVersion)) {
            throw new IllegalStateException("node version [" + joiningNodeVersion + "] may not join a cluster comprising only nodes of version [" + minClusterNodeVersion + "] or greater");
        }
    }

    public static Collection<BiConsumer<DiscoveryNode, ClusterState>> addBuiltInJoinValidators(Collection<BiConsumer<DiscoveryNode, ClusterState>> onJoinValidators) {
        ArrayList<BiConsumer<DiscoveryNode, ClusterState>> validators = new ArrayList<BiConsumer<DiscoveryNode, ClusterState>>();
        validators.add((node, state) -> {
            JoinTaskExecutor.ensureNodesCompatibility(node.getVersion(), state.getNodes());
            JoinTaskExecutor.ensureIndexCompatibility(node.getVersion(), state.getMetadata());
        });
        validators.addAll(onJoinValidators);
        return Collections.unmodifiableCollection(validators);
    }

    public static class Task {
        private final DiscoveryNode node;
        private final String reason;
        private static final String BECOME_MASTER_TASK_REASON = "_BECOME_MASTER_TASK_";
        private static final String FINISH_ELECTION_TASK_REASON = "_FINISH_ELECTION_";

        public Task(DiscoveryNode node, String reason) {
            this.node = node;
            this.reason = reason;
        }

        public DiscoveryNode node() {
            return this.node;
        }

        public String reason() {
            return this.reason;
        }

        public String toString() {
            if (this.node == null) {
                return this.reason;
            }
            StringBuilder stringBuilder = new StringBuilder();
            this.node.appendDescriptionWithoutAttributes(stringBuilder);
            stringBuilder.append(' ').append(this.reason);
            return stringBuilder.toString();
        }

        public boolean isBecomeMasterTask() {
            return this.reason.equals(BECOME_MASTER_TASK_REASON);
        }

        public boolean isFinishElectionTask() {
            return this.reason.equals(FINISH_ELECTION_TASK_REASON);
        }
    }
}

