/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.blobstore.url.http;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.NoSuchFileException;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.common.blobstore.url.http.HttpResponseInputStream;
import org.elasticsearch.common.blobstore.url.http.URLHttpClient;
import org.elasticsearch.common.blobstore.url.http.URLHttpClientException;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.rest.RestStatus;

class RetryingHttpInputStream
extends InputStream {
    public static final int MAX_SUPPRESSED_EXCEPTIONS = 10;
    public static final long MAX_RANGE_VAL = 0x7FFFFFFFFFFFFFFEL;
    private final Logger logger = LogManager.getLogger(RetryingHttpInputStream.class);
    private final String blobName;
    private final URI blobURI;
    private final long start;
    private final long end;
    private final int maxRetries;
    private final URLHttpClient httpClient;
    private long totalBytesRead = 0L;
    private long currentStreamLastOffset = 0L;
    private int retryCount = 0;
    private boolean eof = false;
    private boolean closed = false;
    private HttpResponseInputStream delegate;
    private List<Exception> failures;

    RetryingHttpInputStream(String blobName, URI blobURI, URLHttpClient httpClient, int maxRetries) {
        this(blobName, blobURI, 0L, 0x7FFFFFFFFFFFFFFEL, httpClient, maxRetries);
    }

    RetryingHttpInputStream(String blobName, URI blobURI, long start, long end, URLHttpClient httpClient, int maxRetries) {
        if (start < 0L) {
            throw new IllegalArgumentException("start must be non-negative");
        }
        if (end < start || end == Long.MAX_VALUE) {
            throw new IllegalArgumentException("end must be >= start and not Long.MAX_VALUE");
        }
        this.blobName = blobName;
        this.blobURI = blobURI;
        this.start = start;
        this.end = end;
        this.httpClient = httpClient;
        this.maxRetries = maxRetries;
        this.totalBytesRead = 0L;
        this.retryCount = 0;
    }

    @Override
    public int read() throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                this.maybeOpenInputStream();
                int bytesRead = this.delegate.read();
                if (bytesRead == -1) {
                    this.eof = true;
                    return -1;
                }
                this.totalBytesRead += (long)bytesRead;
                return bytesRead;
            }
            catch (IOException e) {
                this.maybeThrow(e);
                continue;
            }
            break;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                this.maybeOpenInputStream();
                int bytesRead = this.delegate.read(b, off, len);
                if (bytesRead == -1) {
                    this.eof = true;
                    return -1;
                }
                this.totalBytesRead += (long)bytesRead;
                return bytesRead;
            }
            catch (IOException e) {
                this.maybeThrow(e);
                continue;
            }
            break;
        }
    }

    @Override
    public long skip(long n) {
        throw new UnsupportedOperationException("RetryingHttpInputStream does not support seeking");
    }

    @Override
    public void reset() {
        throw new UnsupportedOperationException("RetryingHttpInputStream does not support seeking");
    }

    @Override
    public void close() throws IOException {
        this.maybeAbort(this.delegate);
        try {
            if (this.delegate != null) {
                this.delegate.close();
            }
        }
        finally {
            this.closed = true;
        }
    }

    private void maybeOpenInputStream() throws IOException {
        if (this.delegate == null) {
            this.delegate = this.openInputStream();
        }
    }

    private void ensureOpen() {
        if (this.closed) {
            throw new IllegalStateException("Stream already closed");
        }
    }

    private void maybeThrow(IOException e) throws IOException {
        if (this.retryCount >= this.maxRetries || e instanceof NoSuchFileException) {
            this.logger.debug((Object)new ParameterizedMessage("failed reading [{}] at offset [{}], retry [{}] of [{}], giving up", new Object[]{this.blobURI, this.start + this.totalBytesRead, this.retryCount, this.maxRetries}), (Throwable)e);
            throw this.addSuppressedFailures(e);
        }
        this.logger.debug((Object)new ParameterizedMessage("failed reading [{}] at offset [{}], retry [{}] of [{}], retrying", new Object[]{this.blobURI, this.start + this.totalBytesRead, this.retryCount, this.maxRetries}), (Throwable)e);
        ++this.retryCount;
        this.accumulateFailure(e);
        this.maybeAbort(this.delegate);
        IOUtils.closeWhileHandlingException((Closeable)this.delegate);
        this.delegate = null;
    }

    void maybeAbort(HttpResponseInputStream inputStream) {
        if (this.eof || inputStream == null) {
            return;
        }
        try {
            if (this.start + this.totalBytesRead < this.currentStreamLastOffset) {
                inputStream.abort();
            }
        }
        catch (Exception e) {
            this.logger.warn((Object)"Failed to abort stream before closing", (Throwable)e);
        }
    }

    private void accumulateFailure(Exception e) {
        if (this.failures == null) {
            this.failures = new ArrayList<Exception>(10);
        }
        if (this.failures.size() < 10) {
            this.failures.add(e);
        }
    }

    private IOException addSuppressedFailures(IOException e) {
        if (this.failures == null) {
            return e;
        }
        for (Exception failure : this.failures) {
            e.addSuppressed(failure);
        }
        return e;
    }

    private HttpResponseInputStream openInputStream() throws IOException {
        try {
            return AccessController.doPrivileged(() -> {
                HashMap<String, String> headers = new HashMap<String, String>(1);
                if (this.isRangeRead()) {
                    headers.put("Range", RetryingHttpInputStream.getBytesRange(Math.addExact(this.start, this.totalBytesRead), this.end));
                }
                try {
                    URLHttpClient.HttpResponse response = this.httpClient.get(this.blobURI, headers);
                    int statusCode = response.getStatusCode();
                    if (statusCode != RestStatus.OK.getStatus() && statusCode != RestStatus.PARTIAL_CONTENT.getStatus()) {
                        String body = response.getBodyAsString(1024);
                        IOUtils.closeWhileHandlingException((Closeable)response);
                        throw new IOException(this.getErrorMessage("The server returned an invalid response: Status code: [" + statusCode + "] - Body: " + body));
                    }
                    this.currentStreamLastOffset = Math.addExact(Math.addExact(this.start, this.totalBytesRead), this.getStreamLength(response));
                    return response.getInputStream();
                }
                catch (URLHttpClientException e) {
                    if (e.getStatusCode() == RestStatus.NOT_FOUND.getStatus()) {
                        throw new NoSuchFileException("blob object [" + this.blobName + "] not found");
                    }
                    throw e;
                }
            });
        }
        catch (PrivilegedActionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException)cause;
            }
            throw new IOException(this.getErrorMessage(), e);
        }
        catch (Exception e) {
            throw new IOException(this.getErrorMessage(), e);
        }
    }

    private boolean isRangeRead() {
        return this.start > 0L || this.totalBytesRead > 0L || this.end < 0x7FFFFFFFFFFFFFFEL;
    }

    private long getStreamLength(URLHttpClient.HttpResponse httpResponse) {
        try {
            String contentRange = httpResponse.getHeader("Content-Range");
            if (contentRange != null) {
                Object[] contentRangeTokens = contentRange.split("[ -/]+");
                assert (contentRangeTokens.length == 4) : "Unexpected Content-Range header " + Arrays.toString(contentRangeTokens);
                long lowerBound = Long.parseLong(contentRangeTokens[1]);
                long upperBound = Long.parseLong((String)contentRangeTokens[2]);
                assert (upperBound >= lowerBound) : "Incorrect Content-Range: lower bound > upper bound " + lowerBound + "-" + upperBound;
                assert (lowerBound == this.start + this.totalBytesRead) : "Incorrect Content-Range: lower bound != specified lower bound";
                assert (upperBound == this.end || upperBound <= 0x7FFFFFFFFFFFFFFEL) : "Incorrect Content-Range: the returned upper bound is incorrect, expected [" + this.end + "] got [" + upperBound + "]";
                return upperBound - lowerBound + 1L;
            }
            String contentLength = httpResponse.getHeader("Content-Length");
            return contentLength == null ? 0L : Long.parseLong(contentLength);
        }
        catch (Exception e) {
            this.logger.debug((Object)new ParameterizedMessage("Unable to parse response headers while reading [{}]", (Object)this.blobURI), (Throwable)e);
            return 0x7FFFFFFFFFFFFFFEL;
        }
    }

    private static String getBytesRange(long lowerBound, long upperInclusiveBound) {
        return "bytes=" + lowerBound + "-" + upperInclusiveBound;
    }

    private String getErrorMessage() {
        return this.getErrorMessage("");
    }

    private String getErrorMessage(String extraInformation) {
        String errorMessage = "Unable to read blob [" + this.blobName + "]";
        if (this.isRangeRead()) {
            errorMessage = errorMessage + " range[" + this.start + " - " + this.end + "]";
        }
        if (!extraInformation.isBlank()) {
            errorMessage = errorMessage + " " + extraInformation;
        }
        return errorMessage;
    }
}

