/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.monitor.jvm;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;
import java.time.Clock;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.ToLongFunction;
import org.apache.logging.log4j.LogManager;
import org.apache.lucene.util.CollectionUtil;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.common.time.DateFormatter;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.monitor.jvm.SunThreadInfo;

public class HotThreads {
    private static final Object mutex = new Object();
    private static final StackTraceElement[] EMPTY = new StackTraceElement[0];
    private static final DateFormatter DATE_TIME_FORMATTER = DateFormatter.forPattern("date_optional_time");
    private static final long INVALID_TIMING = -1L;
    private int busiestThreads = 3;
    private TimeValue interval = new TimeValue(500L, TimeUnit.MILLISECONDS);
    private TimeValue threadElementsSnapshotDelay = new TimeValue(10L, TimeUnit.MILLISECONDS);
    private int threadElementsSnapshotCount = 10;
    private ReportType type = ReportType.CPU;
    private SortOrder sortOrder = SortOrder.TOTAL;
    private boolean ignoreIdleThreads = true;
    private static final List<String[]> knownIdleStackFrames = Arrays.asList({"java.util.concurrent.ThreadPoolExecutor", "getTask"}, {"sun.nio.ch.SelectorImpl", "select"}, {"org.elasticsearch.threadpool.ThreadPool$CachedTimeThread", "run"}, {"org.elasticsearch.indices.ttl.IndicesTTLService$Notifier", "await"}, {"java.util.concurrent.LinkedTransferQueue", "poll"}, {"com.sun.jmx.remote.internal.ServerCommunicatorAdmin$Timeout", "run"});
    private static final List<String> knownJDKInternalThreads = Arrays.asList("Signal Dispatcher", "Finalizer", "Reference Handler", "Notification Thread", "Common-Cleaner", "process reaper", "DestroyJavaVM");

    public HotThreads interval(TimeValue interval) {
        this.interval = interval;
        return this;
    }

    public HotThreads busiestThreads(int busiestThreads) {
        this.busiestThreads = busiestThreads;
        return this;
    }

    public HotThreads ignoreIdleThreads(boolean ignoreIdleThreads) {
        this.ignoreIdleThreads = ignoreIdleThreads;
        return this;
    }

    public HotThreads threadElementsSnapshotDelay(TimeValue threadElementsSnapshotDelay) {
        this.threadElementsSnapshotDelay = threadElementsSnapshotDelay;
        return this;
    }

    public HotThreads threadElementsSnapshotCount(int threadElementsSnapshotCount) {
        this.threadElementsSnapshotCount = threadElementsSnapshotCount;
        return this;
    }

    public HotThreads type(ReportType type) {
        this.type = type;
        return this;
    }

    public HotThreads sortOrder(SortOrder order) {
        this.sortOrder = order;
        return this;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String detect() throws Exception {
        Object object = mutex;
        synchronized (object) {
            return this.innerDetect(ManagementFactory.getThreadMXBean(), SunThreadInfo.INSTANCE, Thread.currentThread().getId(), interval -> {
                Thread.sleep(interval);
                return null;
            });
        }
    }

    static boolean isKnownJDKThread(ThreadInfo threadInfo) {
        return knownJDKInternalThreads.stream().anyMatch(jvmThread -> threadInfo.getThreadName() != null && threadInfo.getThreadName().equals(jvmThread));
    }

    static boolean isKnownIdleStackFrame(String className, String methodName) {
        return knownIdleStackFrames.stream().anyMatch(pair -> pair[0].equals(className) && pair[1].equals(methodName));
    }

    static boolean isIdleThread(ThreadInfo threadInfo) {
        if (HotThreads.isKnownJDKThread(threadInfo)) {
            return true;
        }
        for (StackTraceElement frame : threadInfo.getStackTrace()) {
            if (!HotThreads.isKnownIdleStackFrame(frame.getClassName(), frame.getMethodName())) continue;
            return true;
        }
        return false;
    }

    Map<Long, ThreadTimeAccumulator> getAllValidThreadInfos(ThreadMXBean threadBean, SunThreadInfo sunThreadInfo, long currentThreadId) {
        long[] threadIds = threadBean.getAllThreadIds();
        ThreadInfo[] threadInfos = threadBean.getThreadInfo(threadIds);
        HashMap<Long, ThreadTimeAccumulator> result = new HashMap<Long, ThreadTimeAccumulator>(threadIds.length);
        for (int i = 0; i < threadIds.length; ++i) {
            long cpuTime;
            if (threadInfos[i] == null || threadIds[i] == currentThreadId || (cpuTime = threadBean.getThreadCpuTime(threadIds[i])) == -1L) continue;
            long allocatedBytes = this.type == ReportType.MEM ? sunThreadInfo.getThreadAllocatedBytes(threadIds[i]) : 0L;
            result.put(threadIds[i], new ThreadTimeAccumulator(threadInfos[i], this.interval, cpuTime, allocatedBytes));
        }
        return result;
    }

    ThreadInfo[][] captureThreadStacks(ThreadMXBean threadBean, long[] threadIds) throws InterruptedException {
        ThreadInfo[][] result = new ThreadInfo[this.threadElementsSnapshotCount][];
        for (int j = 0; j < this.threadElementsSnapshotCount; ++j) {
            result[j] = threadBean.getThreadInfo(threadIds, Integer.MAX_VALUE);
            Thread.sleep(this.threadElementsSnapshotDelay.millis());
        }
        return result;
    }

    private boolean isThreadWaitBlockTimeMonitoringEnabled(ThreadMXBean threadBean) {
        if (threadBean.isThreadContentionMonitoringSupported()) {
            return threadBean.isThreadContentionMonitoringEnabled();
        }
        return false;
    }

    private double getTimeSharePercentage(long time) {
        return (double)time / (double)this.interval.nanos() * 100.0;
    }

    String innerDetect(ThreadMXBean threadBean, SunThreadInfo sunThreadInfo, long currentThreadId, SleepFunction<Long, Void> threadSleep) throws Exception {
        if (!threadBean.isThreadCpuTimeSupported()) {
            throw new ElasticsearchException("thread CPU time is not supported on this JDK", new Object[0]);
        }
        if (this.type == ReportType.MEM && !sunThreadInfo.isThreadAllocatedMemorySupported()) {
            throw new ElasticsearchException("thread allocated memory is not supported on this JDK", new Object[0]);
        }
        if (!this.isThreadWaitBlockTimeMonitoringEnabled(threadBean)) {
            throw new ElasticsearchException("thread wait/blocked time accounting is not supported on this JDK", new Object[0]);
        }
        StringBuilder sb = new StringBuilder().append("Hot threads at ").append(DATE_TIME_FORMATTER.format(LocalDateTime.now(Clock.systemUTC()))).append(", interval=").append(this.interval).append(", busiestThreads=").append(this.busiestThreads).append(", ignoreIdleThreads=").append(this.ignoreIdleThreads).append(":\n");
        Map<Long, ThreadTimeAccumulator> previousThreadInfos = this.getAllValidThreadInfos(threadBean, sunThreadInfo, currentThreadId);
        threadSleep.apply(this.interval.millis());
        Map<Long, ThreadTimeAccumulator> latestThreadInfos = this.getAllValidThreadInfos(threadBean, sunThreadInfo, currentThreadId);
        latestThreadInfos.forEach((threadId, accumulator) -> accumulator.subtractPrevious((ThreadTimeAccumulator)previousThreadInfos.get(threadId)));
        List<ThreadTimeAccumulator> topThreads = new ArrayList<ThreadTimeAccumulator>(latestThreadInfos.values());
        if (this.type == ReportType.CPU && this.sortOrder == SortOrder.TOTAL) {
            CollectionUtil.introSort(topThreads, Comparator.comparingLong(ThreadTimeAccumulator::getRunnableTime).thenComparingLong(ThreadTimeAccumulator::getCpuTime).reversed());
        } else {
            CollectionUtil.introSort(topThreads, Comparator.comparingLong(ThreadTimeAccumulator.valueGetterForReportType(this.type)).reversed());
        }
        topThreads = topThreads.subList(0, Math.min(this.busiestThreads, topThreads.size()));
        long[] topThreadIds = topThreads.stream().mapToLong(t -> t.threadId).toArray();
        ThreadInfo[][] allInfos = this.captureThreadStacks(threadBean, topThreadIds);
        for (int t2 = 0; t2 < topThreads.size(); ++t2) {
            String threadName = null;
            for (ThreadInfo[] info : allInfos) {
                if (info == null || info[t2] == null) continue;
                if (this.ignoreIdleThreads && HotThreads.isIdleThread(info[t2])) {
                    info[t2] = null;
                    continue;
                }
                threadName = info[t2].getThreadName();
                break;
            }
            if (threadName == null) continue;
            ThreadTimeAccumulator topThread = topThreads.get(t2);
            switch (this.type) {
                case MEM: {
                    sb.append(String.format(Locale.ROOT, "%n%s memory allocated by thread '%s'%n", new ByteSizeValue(topThread.getAllocatedBytes()), threadName));
                    break;
                }
                case CPU: {
                    double percentCpu = this.getTimeSharePercentage(topThread.getCpuTime());
                    double percentOther = this.getTimeSharePercentage(topThread.getOtherTime());
                    sb.append(String.format(Locale.ROOT, "%n%4.1f%% [cpu=%1.1f%%, other=%1.1f%%] (%s out of %s) %s usage by thread '%s'%n", percentOther + percentCpu, percentCpu, percentOther, TimeValue.timeValueNanos((long)(topThread.getCpuTime() + topThread.getOtherTime())), this.interval, this.type.getTypeValue(), threadName));
                    break;
                }
                default: {
                    long time = ThreadTimeAccumulator.valueGetterForReportType(this.type).applyAsLong(topThread);
                    double percent = this.getTimeSharePercentage(time);
                    sb.append(String.format(Locale.ROOT, "%n%4.1f%% (%s out of %s) %s usage by thread '%s'%n", percent, TimeValue.timeValueNanos((long)time), this.interval, this.type.getTypeValue(), threadName));
                }
            }
            boolean[] done = new boolean[this.threadElementsSnapshotCount];
            for (int i = 0; i < this.threadElementsSnapshotCount; ++i) {
                if (done[i]) continue;
                int maxSim = 1;
                boolean[] similars = new boolean[this.threadElementsSnapshotCount];
                for (int j = i + 1; j < this.threadElementsSnapshotCount; ++j) {
                    if (done[j]) continue;
                    int similarity = this.similarity(allInfos[i][t2], allInfos[j][t2]);
                    if (similarity > maxSim) {
                        maxSim = similarity;
                        similars = new boolean[this.threadElementsSnapshotCount];
                    }
                    if (similarity != maxSim) continue;
                    similars[j] = true;
                }
                int count = 1;
                for (int j = i + 1; j < this.threadElementsSnapshotCount; ++j) {
                    if (!similars[j]) continue;
                    done[j] = true;
                    ++count;
                }
                if (allInfos[i][t2] == null) continue;
                StackTraceElement[] show = allInfos[i][t2].getStackTrace();
                if (count == 1) {
                    sb.append(String.format(Locale.ROOT, "  unique snapshot%n", new Object[0]));
                    for (StackTraceElement frame : show) {
                        sb.append(String.format(Locale.ROOT, "    %s%n", frame));
                    }
                    continue;
                }
                sb.append(String.format(Locale.ROOT, "  %d/%d snapshots sharing following %d elements%n", count, this.threadElementsSnapshotCount, maxSim));
                for (int l = show.length - maxSim; l < show.length; ++l) {
                    sb.append(String.format(Locale.ROOT, "    %s%n", show[l]));
                }
            }
        }
        return sb.toString();
    }

    int similarity(ThreadInfo threadInfo, ThreadInfo threadInfo0) {
        StackTraceElement[] s1 = threadInfo == null ? EMPTY : threadInfo.getStackTrace();
        StackTraceElement[] s2 = threadInfo0 == null ? EMPTY : threadInfo0.getStackTrace();
        int i = s1.length - 1;
        int rslt = 0;
        for (int j = s2.length - 1; i >= 0 && j >= 0 && s1[i].equals(s2[j]); --i, --j) {
            ++rslt;
        }
        return rslt;
    }

    public static void initializeRuntimeMonitoring() {
        if (!ManagementFactory.getThreadMXBean().isThreadContentionMonitoringSupported()) {
            LogManager.getLogger(HotThreads.class).info("Thread wait/blocked time accounting not supported.");
        } else {
            try {
                ManagementFactory.getThreadMXBean().setThreadContentionMonitoringEnabled(true);
            }
            catch (UnsupportedOperationException monitoringUnavailable) {
                LogManager.getLogger(HotThreads.class).warn("Thread wait/blocked time accounting cannot be enabled.");
            }
        }
    }

    public static enum ReportType {
        CPU("cpu"),
        WAIT("wait"),
        BLOCK("block"),
        MEM("mem");

        private final String type;

        private ReportType(String type) {
            this.type = type;
        }

        public String getTypeValue() {
            return this.type;
        }

        public static ReportType of(String type) {
            for (ReportType report : ReportType.values()) {
                if (!report.type.equals(type)) continue;
                return report;
            }
            throw new IllegalArgumentException("type not supported [" + type + "]");
        }
    }

    public static enum SortOrder {
        TOTAL("total"),
        CPU("cpu");

        private final String order;

        private SortOrder(String order) {
            this.order = order;
        }

        public String getOrderValue() {
            return this.order;
        }

        public static SortOrder of(String order) {
            for (SortOrder sortOrder : SortOrder.values()) {
                if (!sortOrder.order.equals(order)) continue;
                return sortOrder;
            }
            throw new IllegalArgumentException("sort order not supported [" + order + "]");
        }
    }

    @FunctionalInterface
    public static interface SleepFunction<T, R> {
        public R apply(T var1) throws InterruptedException;
    }

    static class ThreadTimeAccumulator {
        private final long threadId;
        private final TimeValue interval;
        private long cpuTime;
        private long blockedTime;
        private long waitedTime;
        private long allocatedBytes;

        ThreadTimeAccumulator(ThreadInfo info, TimeValue interval, long cpuTime, long allocatedBytes) {
            this.blockedTime = this.millisecondsToNanos(info.getBlockedTime());
            this.waitedTime = this.millisecondsToNanos(info.getWaitedTime());
            this.cpuTime = cpuTime;
            this.allocatedBytes = allocatedBytes;
            this.threadId = info.getThreadId();
            this.interval = interval;
        }

        private long millisecondsToNanos(long millis) {
            return millis * 1000000L;
        }

        void subtractPrevious(ThreadTimeAccumulator previous) {
            if (previous != null) {
                if (previous.getThreadId() != this.getThreadId()) {
                    throw new IllegalStateException("Thread timing accumulation must be done on the same thread");
                }
                this.blockedTime -= previous.blockedTime;
                this.waitedTime -= previous.waitedTime;
                this.cpuTime -= previous.cpuTime;
                this.allocatedBytes -= previous.allocatedBytes;
            }
        }

        public long getCpuTime() {
            return Math.max(this.cpuTime, 0L);
        }

        public long getRunnableTime() {
            if (this.getCpuTime() == 0L) {
                return 0L;
            }
            return Math.max(this.interval.nanos() - this.getWaitedTime() - this.getBlockedTime(), 0L);
        }

        public long getOtherTime() {
            if (this.getCpuTime() == 0L) {
                return 0L;
            }
            return Math.max(this.getRunnableTime() - this.getCpuTime(), 0L);
        }

        public long getBlockedTime() {
            return Math.max(this.blockedTime, 0L);
        }

        public long getWaitedTime() {
            return Math.max(this.waitedTime, 0L);
        }

        public long getAllocatedBytes() {
            return Math.max(this.allocatedBytes, 0L);
        }

        public long getThreadId() {
            return this.threadId;
        }

        static ToLongFunction<ThreadTimeAccumulator> valueGetterForReportType(ReportType type) {
            switch (type) {
                case CPU: {
                    return ThreadTimeAccumulator::getCpuTime;
                }
                case WAIT: {
                    return ThreadTimeAccumulator::getWaitedTime;
                }
                case BLOCK: {
                    return ThreadTimeAccumulator::getBlockedTime;
                }
                case MEM: {
                    return ThreadTimeAccumulator::getAllocatedBytes;
                }
            }
            throw new IllegalArgumentException("expected thread type to be either 'cpu', 'wait', 'mem', or 'block', but was " + type);
        }
    }
}

